<?php

namespace IQDEV\ElasticSearchTests\Filter;

use IQDEV\ElasticSearchTests\AbstractTestCase;
use IQDEV\ElasticSearchTests\Helpers\FormatData;
use IQDEV\ElasticSearchTests\Service\SearchClient;
use IQDEV\ElasticSearch\Criteria;
use IQDEV\ElasticSearch\Filter\Field;
use IQDEV\ElasticSearch\Filter\Filter;
use IQDEV\ElasticSearch\Filter\FilterGroupCollection;
use IQDEV\ElasticSearch\Filter\FilterKeyword;
use IQDEV\ElasticSearch\Filter\FilterNumber;
use IQDEV\ElasticSearch\Filter\FilterOperator;
use IQDEV\ElasticSearch\Filter\FilterType;
use IQDEV\ElasticSearch\Filter\LogicOperator;
use IQDEV\ElasticSearch\Query\SearchQuery;

class QueryAndPostFilterTest extends AbstractTestCase
{
    /**
     * query по не сущ бренду
     */
    public function testQueryNonExistentBrand()
    {
        $criteria = new Criteria();

        $filterCollectionBrand = new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('not')
            )
        ]);
        $filterCollectionBrand->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionBrand);


        $q = new SearchQuery($criteria);
        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [],
            'facets' => []
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    /**
     * query по сущ бренду
     * @return void
     */
    public function testQueryExistingBrand()
    {
        $criteria = new Criteria();

        $filterCollectionBrand = new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('adidas')
            )
        ]);
        $filterCollectionBrand->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionBrand);


        $q = new SearchQuery($criteria);
        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                's1',
                's2',
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'adidas',
                                'count' => 2,
                                'active' => true,
                            ]
                        ],
                        'range' => [],
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'green',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => [],
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => '46',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => '47',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => [],
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 2,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 100.0,
                                    'max' => 101.0,
                                ],
                                'activeRange' => [
                                    'min' => 100.0,
                                    'max' => 101.0,
                                ]
                            ]
                        ]
                    ]
                ]
            ],
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    /**
     * query по сущ бренду и post_filter по бренду
     * @return void
     */
    public function testQueryExistingBrandAndPostFilterExistingBrand()
    {
        $criteria = new Criteria();

        $filterCollectionBrand = new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('adidas')
            ),
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('nike')
            ),
        ]);
        $filterCollectionBrand
            ->setLogicalType(LogicOperator::or())
            ->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionBrand);

        $filterCollectionBrand = new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('nike')
            ),
        ]);
        $criteria->filters()->add($filterCollectionBrand);


        $q = new SearchQuery($criteria);
        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                's4',
                'h1',
                'h2',
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'adidas',
                                'count' => 2,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 3,
                                'active' => true,
                            ]
                        ],
                        'range' => [],
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'green',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true,
                            ],
                            2 => [
                                'label' => null,
                                'value' => 'white',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => [],
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => '43',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => '46',
                                'count' => 0,
                                'active' => false,
                            ],
                            2 => [
                                'label' => null,
                                'value' => '47',
                                'count' => 0,
                                'active' => false,
                            ],
                            3 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 1,
                                'active' => true,
                            ],
                            4 => [
                                'label' => null,
                                'value' => 'xxl',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => [],
                    ],
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 3,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 100.0,
                                    'max' => 105.0,
                                ],
                                'activeRange' => [
                                    'min' => 103.0,
                                    'max' => 105.0,
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    /**
     * query по сущ бренду и post_filter по цвету
     * @return void
     */
    public function testQueryExistingBrandAndPostFilterExistingColor()
    {
        $criteria = new Criteria();

        $filterCollectionBrand = new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('nike')
            ),
        ]);
        $filterCollectionBrand
            ->setLogicalType(LogicOperator::or())
            ->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionBrand);

        $filterCollectionColor = new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('white')
            ),
        ]);
        $criteria->filters()->add($filterCollectionColor);


        $q = new SearchQuery($criteria);
        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                'h2',
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => [],
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'green',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true,
                            ],
                            2 => [
                                'label' => null,
                                'value' => 'white',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => [],
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => '43',
                                'count' => 0,
                                'active' => false,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 0,
                                'active' => false,
                            ],
                            2 => [
                                'label' => null,
                                'value' => 'xxl',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => [],
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 1,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 103.0,
                                    'max' => 105.0,
                                ],
                                'activeRange' => [
                                    'min' => 105.0,
                                    'max' => 105.0,
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    /**
     * query по сущ бренду и цвету и post_filter по бренду
     * @return void
     */
    public function testQueryExistingBrandAndColorAndPostFilterExistingColor()
    {
        $criteria = new Criteria();

        $filterCollectionBrand = new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('nike')
            ),
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('rebook')
            ),
        ]);
        $filterCollectionBrand
            ->setLogicalType(LogicOperator::or())
            ->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionBrand);

        $filterCollectionColor = new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('white')
            ),
        ]);
        $filterCollectionColor->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionColor);

        $filterCollectionBrand = new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('nike')
            ),
        ]);
        $criteria->filters()->add($filterCollectionBrand);


        $q = new SearchQuery($criteria);
        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                'h2',
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'rebook',
                                'count' => 2,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'white',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 0,
                                'active' => false,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'xxl',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 1,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 105.0,
                                    'max' => 107.0,
                                ],
                                'activeRange' => [
                                    'min' => 105.0,
                                    'max' => 105.0,
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    /**
     * query по сущ бренду и post_filter по бренду и цвету
     * @return void
     */
    public function testQueryExistingBrandAndPostFilterExistingBrandAndColor()
    {
        $criteria = new Criteria();

        $filterCollectionBrand = new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('nike')
            ),
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('rebook')
            ),
        ]);
        $filterCollectionBrand
            ->setLogicalType(LogicOperator::or())
            ->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionBrand);


        $filterCollectionBrand = new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('nike')
            ),
        ]);
        $criteria->filters()->add($filterCollectionBrand);

        $filterCollectionColor = new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('red')
            ),
        ]);
        $criteria->filters()->add($filterCollectionColor);


        $q = new SearchQuery($criteria);
        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                'h1',
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'rebook',
                                'count' => 0,
                                'active' => false,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'blue',
                                'count' => 0,
                                'active' => false,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'green',
                                'count' => 1,
                                'active' => true,
                            ],
                            2 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true,
                            ],
                            3 => [
                                'label' => null,
                                'value' => 'white',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => '43',
                                'count' => 0,
                                'active' => false,
                            ],
                            1 => [
                                'label' => null,
                                'value' => '47',
                                'count' => 0,
                                'active' => false,
                            ],
                            2 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 1,
                                'active' => true,
                            ],
                            3 => [
                                'label' => null,
                                'value' => 'xxl',
                                'count' => 0,
                                'active' => false,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 1,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 102.0,
                                    'max' => 107.0,
                                ],
                                'activeRange' => [
                                    'min' => 104.0,
                                    'max' => 104.0,
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    /**
     * query по не сущ цене
     * @return void
     */
    public function testQueryNonExistentPrice()
    {
        $criteria = new Criteria();

        $filterCollectionPrice = new FilterGroupCollection([
            new Filter(
                new Field('price'),
                new FilterOperator(FilterOperator::GTE),
                new FilterNumber(108)
            )
        ]);
        $filterCollectionPrice->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionPrice);


        $q = new SearchQuery($criteria);
        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [],
            'facets' => [],
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    /**
     * query по сущ цене
     * @return void
     */
    public function testQueryExistingPrice()
    {
        $criteria = new Criteria();

        $filterCollectionPrice = new FilterGroupCollection([
            new Filter(
                new Field('price'),
                new FilterOperator(FilterOperator::GTE),
                new FilterNumber(105)
            )
        ]);
        $filterCollectionPrice->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionPrice);


        $q = new SearchQuery($criteria);
        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                'h2',
                'h3',
                'p1',
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'rebook',
                                'count' => 2,
                                'active' => true,
                            ]
                        ],
                        'range' => [],
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'white',
                                'count' => 3,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 2,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'xxl',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 3,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 105.0,
                                    'max' => 107.0,
                                ],
                                'activeRange' => [
                                    'min' => 105.0,
                                    'max' => 107.0,
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    /**
     * query по сущ цене и бренду
     * @return void
     */
    public function testQueryExistingPriceAndBrand()
    {
        $criteria = new Criteria();

        $filterCollection = new FilterGroupCollection([
            new Filter(
                new Field('price'),
                new FilterOperator(FilterOperator::GTE),
                new FilterNumber(104)
            ),
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('nike')
            ),
        ]);
        $filterCollection->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollection);

        $q = new SearchQuery($criteria);
        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                'h1',
                'h2',
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 2,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'white',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'xxl',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 2,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 104.0,
                                    'max' => 105.0,
                                ],
                                'activeRange' => [
                                    'min' => 104.0,
                                    'max' => 105.0,
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    /**
     * query по сущ цене и бренду и post_filter по цвету
     * @return void
     */
    public function testQueryExistingPriceAndBrandAndPostFilterColor()
    {
        $criteria = new Criteria();

        $filterCollection = new FilterGroupCollection([
            new Filter(
                new Field('price'),
                new FilterOperator(FilterOperator::GTE),
                new FilterNumber(104)
            ),
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('nike')
            ),
        ]);
        $filterCollection->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollection);

        $filterCollectionColor = new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('red')
            ),
        ]);
        $criteria->filters()->add($filterCollectionColor);

        $q = new SearchQuery($criteria);
        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                'h1',
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'white',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'xxl',
                                'count' => 0,
                                'active' => false,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 1,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 104.0,
                                    'max' => 105.0,
                                ],
                                'activeRange' => [
                                    'min' => 104.0,
                                    'max' => 104.0,
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    /**
     * query по сущ цене и бренду и post_filter по цвету и бренду
     * @return void
     */
    public function testQueryExistingPriceAndBrandAndPostFilterColorAndBrand()
    {
        $criteria = new Criteria();

        $filterCollectionPrice = new FilterGroupCollection([
            new Filter(
                new Field('price'),
                new FilterOperator(FilterOperator::LTE),
                new FilterNumber(104)
            ),
        ]);
        $filterCollectionPrice->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionPrice);

        $filterCollectionBrand = new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('nike')
            ),
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('adidas')
            ),
        ]);
        $filterCollectionBrand
            ->setLogicalType(LogicOperator::or())
            ->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionBrand);


        $filterCollectionColor = new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('green')
            ),
        ]);
        $criteria->filters()->add($filterCollectionColor);

        $filterCollectionBrand = new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('nike')
            ),
        ]);
        $criteria->filters()->add($filterCollectionBrand);


        $q = new SearchQuery($criteria);
        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                's4',
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'adidas',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'green',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => '43',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => '46',
                                'count' => 0,
                                'active' => false,
                            ],
                            2 => [
                                'label' => null,
                                'value' => '47',
                                'count' => 0,
                                'active' => false,
                            ],
                            3 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 0,
                                'active' => false,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 1,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 100.0,
                                    'max' => 104.0,
                                ],
                                'activeRange' => [
                                    'min' => 103.0,
                                    'max' => 103.0,
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    /**
     * query по сущ цене и бренду и post_filter по цене
     * @return void
     */
    public function testQueryExistingPriceAndBrandAndPostFilterPrice()
    {
        $criteria = new Criteria();

        $filterCollectionPrice = new FilterGroupCollection([
            new Filter(
                new Field('price'),
                new FilterOperator(FilterOperator::LTE),
                new FilterNumber(104)
            ),
        ]);
        $filterCollectionPrice->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionPrice);

        $filterCollectionBrand = new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('nike')
            ),
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('adidas')
            ),
        ]);
        $filterCollectionBrand
            ->setLogicalType(LogicOperator::or())
            ->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionBrand);


        $filterCollectionPrice = new FilterGroupCollection([
            new Filter(
                new Field('price'),
                new FilterOperator(FilterOperator::GTE),
                new FilterNumber(102)
            ),
        ]);
        $criteria->filters()->add($filterCollectionPrice);


        $q = new SearchQuery($criteria);
        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                's4',
                'h1',
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'adidas',
                                'count' => 0,
                                'active' => false,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 2,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'green',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => '43',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => '46',
                                'count' => 0,
                                'active' => false,
                            ],
                            2 => [
                                'label' => null,
                                'value' => '47',
                                'count' => 0,
                                'active' => false,
                            ],
                            3 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => [],
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 2,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 100.0,
                                    'max' => 104.0,
                                ],
                                'activeRange' => [
                                    'min' => 103.0,
                                    'max' => 104.0,
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    /**
     * query по сущ цене и бренду и post_filter по цвету и цене
     * @return void
     */
    public function testQueryExistingPriceAndBrandAndPostFilterPriceAndColor()
    {
        $criteria = new Criteria();

        $filterCollectionPrice = new FilterGroupCollection([
            new Filter(
                new Field('price'),
                new FilterOperator(FilterOperator::LTE),
                new FilterNumber(104)
            ),
        ]);
        $filterCollectionPrice->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionPrice);

        $filterCollectionBrand = new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('nike')
            ),
            new Filter(
                new Field('brand'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('adidas')
            ),
        ]);
        $filterCollectionBrand
            ->setLogicalType(LogicOperator::or())
            ->setFilterType(FilterType::query());
        $criteria->filters()->add($filterCollectionBrand);


        $filterCollectionPrice = new FilterGroupCollection([
            new Filter(
                new Field('price'),
                new FilterOperator(FilterOperator::GTE),
                new FilterNumber(101)
            ),
        ]);
        $criteria->filters()->add($filterCollectionPrice);

        $filterCollectionColor = new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('green')
            ),
        ]);
        $criteria->filters()->add($filterCollectionColor);


        $q = new SearchQuery($criteria);
        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                's4',
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'adidas',
                                'count' => 0,
                                'active' => false,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 1,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'green',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 2,
                                'active' => true,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => '43',
                                'count' => 1,
                                'active' => true,
                            ],
                            1 => [
                                'label' => null,
                                'value' => '46',
                                'count' => 0,
                                'active' => false,
                            ],
                            2 => [
                                'label' => null,
                                'value' => '47',
                                'count' => 0,
                                'active' => false,
                            ],
                            3 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 0,
                                'active' => false,
                            ]
                        ],
                        'range' => []
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 1,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 100.0,
                                    'max' => 104.0,
                                ],
                                'activeRange' => [
                                    'min' => 103.0,
                                    'max' => 103.0,
                                ],
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }
}
