<?php

namespace IQDEV\ElasticSearchTests\Filter;

use IQDEV\ElasticSearchTests\AbstractTestCase;
use IQDEV\ElasticSearchTests\Helpers\FormatData;
use IQDEV\ElasticSearchTests\Service\SearchClient;
use IQDEV\ElasticSearch\Criteria;
use IQDEV\ElasticSearch\Filter\Field;
use IQDEV\ElasticSearch\Filter\Filter;
use IQDEV\ElasticSearch\Filter\FilterGroupCollection;
use IQDEV\ElasticSearch\Filter\FilterKeyword;
use IQDEV\ElasticSearch\Filter\FilterNumber;
use IQDEV\ElasticSearch\Filter\FilterOperator;
use IQDEV\ElasticSearch\Query\SearchQuery;

class CommonRangeKeywordsTest extends AbstractTestCase
{
    /**
     * Поиск элементов по фильтру свойства и цене
     *
     * @return void
     */
    public function testExistByFilterAndMinPrice(): void
    {
        $criteria = new Criteria();

        $criteria->filters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('red')
            )
        ]));

        $criteria->filters()->add(new FilterGroupCollection([
            new Filter(
                new Field('price'),
                new FilterOperator(FilterOperator::GT),
                new FilterNumber(102)
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'h1'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    /**
     * Поиск элементов по фильтру свойства и цене
     *
     * @return void
     */
    public function testExistByFilterAndMaxPrice(): void
    {
        $criteria = new Criteria();

        $criteria->filters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('red')
            )
        ]));

        $criteria->filters()->add(new FilterGroupCollection([
            new Filter(
                new Field('price'),
                new FilterOperator(FilterOperator::LT),
                new FilterNumber(102)
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                's2'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    /**
     * Поиск элементов по фильтру свойства и цене
     *
     * @return void
     */
    public function testExistByFilterAndRangePrice(): void
    {
        $criteria = new Criteria();

        $criteria->filters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('red')
            )
        ]));

        $criteria->filters()->add(new FilterGroupCollection([
            new Filter(
                new Field('price'),
                new FilterOperator(FilterOperator::GTE),
                new FilterNumber(101)
            ),
            new Filter(
                new Field('price'),
                new FilterOperator(FilterOperator::LTE),
                new FilterNumber(104)
            ),
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                'h1',
                's2'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }
}