<?php

namespace IQDEV\ElasticSearch\Indexer;

use Elastic\Elasticsearch\Client;
use Elastic\Elasticsearch\Exception\ClientResponseException;
use Elastic\Elasticsearch\Exception\ContentTypeException;
use Elastic\Elasticsearch\Exception\MissingParameterException;
use Elastic\Elasticsearch\Exception\ServerResponseException;
use Elastic\Elasticsearch\Response\Elasticsearch;
use Elastic\Elasticsearch\Traits\EndpointTrait;
use IQDEV\ElasticSearch\Configuration;
use Psr\Log\LoggerInterface;

final class EsHelperEndpoint
{
    use EndpointTrait;

    private Client $esClient;
    private Configuration $configuration;

    private LoggerInterface $logger;

    public function __construct(
        Client          $esClient,
        Configuration   $configuration,
        LoggerInterface $logger
    )
    {
        $this->esClient = $esClient;
        $this->configuration = $configuration;
        $this->logger = $logger;
    }

    public function isIndexExists(): bool
    {
        $response = $this->esClient
            ->indices()
            ->exists(
                [
                    'index' => $this->configuration->getIndexName(),
                ]
            );

        return $response instanceof Elasticsearch && true === $response->asBool();
    }

    /**
     * Создание индекса
     *
     * @return void
     *
     * @throws \Elastic\Elasticsearch\Exception\ClientResponseException
     * @throws \Elastic\Elasticsearch\Exception\MissingParameterException
     * @throws \Elastic\Elasticsearch\Exception\ServerResponseException
     */
    public function create(): void
    {
        if (false === $this->isIndexExists()) {
            $this->logger->info(sprintf('Index %s was created', $this->configuration->getIndexName()));
            $this->esClient->indices()->create(
                [
                    'index' => $this->configuration->getIndexName(),
                    'body' => [
                        'mappings' => $this->configuration->getMapping(),
                        'settings' => $this->configuration->getSettings(),
                    ],
                ]
            );
        }
    }

    /**
     * Обновление конфигурации индекса
     *
     * @throws ContentTypeException
     */
    public function reconfigurate(): void
    {
        $this->esClient->sendRequest(
            $this->createRequest(
                'POST',
                '/' . $this->encode($this->configuration->getIndexName()) . '/_close',
                [
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                [],
            )
        );
        $this->esClient->sendRequest(
            $this->createRequest(
                'PUT',
                '/' . $this->encode($this->configuration->getIndexName()) . '/_settings',
                [
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                $this->configuration->getSettings(),
            )
        );

        $this->esClient->sendRequest(
            $this->createRequest(
                'PUT',
                '/' . $this->encode($this->configuration->getIndexName()) . '/_mapping',
                [
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                $this->configuration->getMapping(),
            )
        );

        $this->esClient->sendRequest(
            $this->createRequest(
                'POST',
                '/' . $this->encode($this->configuration->getIndexName()) . '/_open',
                [
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                [],
            )
        );
        $this->logger->info(sprintf('Index %s was reconfigurated', $this->configuration->getIndexName()));
    }

    /**
     * Полная очистка документов индекса
     *
     * @throws ContentTypeException
     */
    public function clear(): void
    {
        if ($this->isIndexExists()) {
            $this->esClient->sendRequest(
                $this->createRequest(
                    'POST',
                    '/' . $this->encode($this->configuration->getIndexName()) . '/_delete_by_query',
                    [
                        'Accept' => 'application/json',
                        'Content-Type' => 'application/json',
                    ],
                    '{"query": {"match_all": {}}}'
                )
            );
            $this->logger->info(sprintf('Index %s was cleared', $this->configuration->getIndexName()));
        } else {
            $this->logger->error(sprintf('Index %s does not exists', $this->configuration->getIndexName()));
        }
    }

    /**
     * Удаление индекса
     *
     * @return void
     *
     * @throws ClientResponseException
     * @throws MissingParameterException
     * @throws ServerResponseException
     */
    public function delete(): void
    {
        if ($this->isIndexExists()) {
            $response = $this->esClient
                ->indices()
                ->delete(
                    [
                        'index' => $this->configuration->getIndexName(),
                    ]
                );
            if (($response instanceof Elasticsearch) && false === $response->asBool()) {
                $this->logger->info(sprintf('Index %s was deleted', $this->configuration->getIndexName()));
            } else {
                $this->logger->error(sprintf('Index %s was not deleted', $this->configuration->getIndexName()));
            }
        }
    }
}