<?php

namespace IQDEV\ElasticSearchTests\Filter;

use IQDEV\ElasticSearch\Criteria\Criteria;
use IQDEV\ElasticSearch\Criteria\Filter\Collection\FilterGroupCollection;
use IQDEV\ElasticSearch\Criteria\Filter\Field;
use IQDEV\ElasticSearch\Criteria\Filter\Filter;
use IQDEV\ElasticSearch\Criteria\Filter\FilterOperator;
use IQDEV\ElasticSearch\Criteria\Filter\LogicOperator;
use IQDEV\ElasticSearch\Criteria\Filter\Value\FilterKeyword;
use IQDEV\ElasticSearch\Criteria\Filter\Value\FilterNumber;
use IQDEV\ElasticSearch\Criteria\Query\SearchQuery;
use IQDEV\ElasticSearch\Criteria\Search\Search;
use IQDEV\ElasticSearch\Document\Property\Property;
use IQDEV\ElasticSearchTests\AbstractTestCase;
use IQDEV\ElasticSearchTests\Helpers\FormatData;
use IQDEV\ElasticSearchTests\Service\SearchClient;

/**
 * Тестирование агрегирующих функций
 */
class AggsTest extends AbstractTestCase
{
    public function testEmptyFilterByCategory()
    {
        $criteria = new Criteria();
        $criteria->getSearch()->add(
            new Search(
                new Property('category_id'),
                't-short',
            ),
        );

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                'h1',
                'h2',
                'h3'
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 2,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'rebook',
                                'count' => 1,
                                'active' => true
                            ]
                        ],
                        'range' => []
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'white',
                                'count' => 2,
                                'active' => true
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 2,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'xxl',
                                'count' => 1,
                                'active' => true
                            ]
                        ],
                        'range' => []
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 3,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 104.0,
                                    'max' => 106.0
                                ],
                                'activeRange' => [
                                    'min' => 104.0,
                                    'max' => 106.0
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    public function testEmptyKeywordFilter()
    {
        $criteria = new Criteria();
        $criteria->getSearch()->add(
            new Search(
                new Property('category_id'),
                't-short',
            ),
        );

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                FilterOperator::EQ,
                new FilterKeyword('white')
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                'h2',
                'h3'
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'rebook',
                                'count' => 1,
                                'active' => true
                            ]
                        ],
                        'range' => []
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'white',
                                'count' => 2,
                                'active' => true
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'xxl',
                                'count' => 1,
                                'active' => true
                            ]
                        ],
                        'range' => []
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 2,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 104.0,
                                    'max' => 106.0
                                ],
                                'activeRange' => [
                                    'min' => 105.0,
                                    'max' => 106.0
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    public function testRangeFilter()
    {
        $criteria = new Criteria();
        $criteria->getSearch()->add(
            new Search(
                new Property('category_id'),
                't-short',
            ),
        );
        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('price'),
                FilterOperator::LTE,
                new FilterNumber(104.50)
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                'h1',
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'rebook',
                                'count' => 0,
                                'active' => false
                            ]
                        ],
                        'range' => []
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'white',
                                'count' => 0,
                                'active' => false
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'xxl',
                                'count' => 0,
                                'active' => false
                            ]
                        ],
                        'range' => []
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 1,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 104.0,
                                    'max' => 106.0
                                ],
                                'activeRange' => [
                                    'min' => 104.0,
                                    'max' => 104.0
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    public function testCombineFilter()
    {
        $criteria = new Criteria();
        $criteria->getSearch()->add(
            new Search(
                new Property('category_id'),
                't-short',
            ),
        );

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                FilterOperator::EQ,
                new FilterKeyword('black')
            )
        ]));

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('price'),
                FilterOperator::LTE,
                new FilterNumber(104)
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 0,
                                'active' => false
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'rebook',
                                'count' => 0,
                                'active' => false
                            ]
                        ],
                        'range' => []
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'white',
                                'count' => 0,
                                'active' => false
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 0,
                                'active' => false
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'xxl',
                                'count' => 0,
                                'active' => false
                            ]
                        ],
                        'range' => []
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 0,
                                'active' => false,
                                'fullRange' => [
                                    'min' => 104.0,
                                    'max' => 106.0
                                ],
                                'activeRange' => [
                                    'min' => null,
                                    'max' => null
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    public function testCombineFilterTwo()
    {
        $criteria = new Criteria();
        $criteria->getSearch()->add(
            new Search(
                new Property('category_id'),
                't-short',
            ),
        );

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                FilterOperator::EQ,
                new FilterKeyword('red')
            )
        ]));

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('price'),
                FilterOperator::LTE,
                new FilterNumber(105)
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'h1'
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'rebook',
                                'count' => 0,
                                'active' => false
                            ]
                        ],
                        'range' => []
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'white',
                                'count' => 1,
                                'active' => true
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'xxl',
                                'count' => 0,
                                'active' => false
                            ]
                        ],
                        'range' => []
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 1,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 104.0,
                                    'max' => 106.0
                                ],
                                'activeRange' => [
                                    'min' => 104.0,
                                    'max' => 104.0
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    public function testKeywordFilter()
    {
        $criteria = new Criteria();
        $criteria->getSearch()->add(
            new Search(
                new Property('category_id'),
                't-short',
            ),
        );

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                FilterOperator::EQ,
                new FilterKeyword('white')
            )
        ]));

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                FilterOperator::EQ,
                new FilterKeyword('nike')
            )
        ]));

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('price'),
                FilterOperator::GT,
                new FilterNumber(100)
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'h2'
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'rebook',
                                'count' => 1,
                                'active' => true
                            ]
                        ],
                        'range' => []
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'white',
                                'count' => 1,
                                'active' => true
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 0,
                                'active' => false
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'xxl',
                                'count' => 1,
                                'active' => true
                            ]
                        ],
                        'range' => []
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 1,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 104.0,
                                    'max' => 106.0
                                ],
                                'activeRange' => [
                                    'min' => 105.0,
                                    'max' => 105.0
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }

    public function testKeywordFilterTwo()
    {
        $criteria = new Criteria();
        $criteria->getSearch()->add(
            new Search(
                new Property('category_id'),
                't-short',
            ),
        );

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                FilterOperator::EQ,
                new FilterKeyword('white')
            )
        ]));

        $criteria->getFilters()->add(
            (new FilterGroupCollection([
                new Filter(
                    new Field('brand'),
                    FilterOperator::EQ,
                    new FilterKeyword('nike')
                ),
                new Filter(
                    new Field('brand'),
                    FilterOperator::EQ,
                    new FilterKeyword('reebok')
                )
            ]))
                ->setLogicOperator(LogicOperator::OR)
        );


        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'h2'
            ],
            'facets' => [
                0 => [
                    'code' => 'brand',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'nike',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'rebook',
                                'count' => 1,
                                'active' => true
                            ]
                        ],
                        'range' => []
                    ]
                ],
                1 => [
                    'code' => 'color',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'red',
                                'count' => 1,
                                'active' => true
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'white',
                                'count' => 1,
                                'active' => true
                            ]
                        ],
                        'range' => []
                    ]
                ],
                2 => [
                    'code' => 'size',
                    'label' => null,
                    'type' => 'list',
                    'items' => [
                        'list' => [
                            0 => [
                                'label' => null,
                                'value' => 'xl',
                                'count' => 0,
                                'active' => false
                            ],
                            1 => [
                                'label' => null,
                                'value' => 'xxl',
                                'count' => 1,
                                'active' => true
                            ]
                        ],
                        'range' => []
                    ]
                ],
                3 => [
                    'code' => 'price',
                    'label' => null,
                    'type' => 'range',
                    'items' => [
                        'list' => [],
                        'range' => [
                            0 => [
                                'label' => null,
                                'count' => 1,
                                'active' => true,
                                'fullRange' => [
                                    'min' => 104.0,
                                    'max' => 106.0
                                ],
                                'activeRange' => [
                                    'min' => 105.0,
                                    'max' => 105.0
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatDataWFacets($result));
    }
}