<?php

namespace IQDEV\ElasticSearch\Search;

use IQDEV\ElasticSearch\Esable;
use IQDEV\ElasticSearch\Search\Aggs\AggsCollection;
use IQDEV\ElasticSearch\Search\BoolQuery\Query;
use IQDEV\ElasticSearch\Search\Sorting\SortingCollection;

final class Request implements Esable
{
    private ?Query $query = null;
    private ?Query $postFilter = null;
    private ?AggsCollection $aggs = null;
    private ?Pagination $pagination = null;
    private ?SortingCollection $sorting = null;
    private array $match = [];

    /**
     * @param Pagination|null $pagination
     * @return Request
     */
    public function setPagination(?Pagination $pagination): self
    {
        $this->pagination = $pagination;
        return $this;
    }

    /**
     * @param SortingCollection|null $sorting
     * @return $this
     */
    public function setSorting(?SortingCollection $sorting): self
    {
        $this->sorting = $sorting;
        return $this;
    }

    /**
     * @return Query
     */
    public function getQuery(): Query
    {
        if ($this->query === null) {
            $this->query = new Query();
        }

        return $this->query;
    }

    /**
     * @return Query
     */
    public function getPostFilter(): Query
    {
        if ($this->postFilter === null) {
            $this->postFilter = new Query();
        }

        return $this->postFilter;
    }

    public function getAggs(): AggsCollection
    {
        if ($this->aggs === null) {
            $this->aggs = new AggsCollection();
        }

        return $this->aggs;
    }

    /**
     * @return Pagination|null
     */
    public function getPagination(): ?Pagination
    {
        return $this->pagination;
    }

    /**
     * @return SortingCollection|null
     */
    public function getSorting(): ?SortingCollection
    {
        return $this->sorting;
    }

    public function addMatch(string $key, array $param): self
    {
        $this->match[$key] = $param;
        return $this;
    }

    public function es(): array
    {
        $request = [
            '_source' => ['id']
        ];

        if (isset($this->postFilter) && $this->postFilter->isEmpty() === false) {
            $request['post_filter'] = $this->postFilter->es()['query'];
        }

        if (isset($this->query) && $this->query->isEmpty() === false) {
            $request['query'] = $this->query->es()['query'];
        }

        if (empty($this->match) === false) {
            foreach ($this->match as $key => $value) {
                $request['query']['match'][$key] = $value;
            }
        }

        if ($this->aggs) {
            $request['aggs'] = $this->aggs->es()['aggs'];
        }

        if ($this->pagination) {
            $request = array_merge($request, $this->pagination->es());
        }
        
        if ($this->sorting) {
            $request['sort'] = $this->sorting->es();
        }

        return $request;
    }
}