<?php

namespace IQDEV\ElasticSearchTests\Filter;

use IQDEV\ElasticSearch\Criteria\Criteria;
use IQDEV\ElasticSearch\Criteria\Filter\Collection\FilterGroupCollection;
use IQDEV\ElasticSearch\Criteria\Filter\Field;
use IQDEV\ElasticSearch\Criteria\Filter\Filter;
use IQDEV\ElasticSearch\Criteria\Filter\FilterOperator;
use IQDEV\ElasticSearch\Criteria\Filter\Value\FilterKeyword;
use IQDEV\ElasticSearch\Criteria\Query\SearchQuery;
use IQDEV\ElasticSearch\Criteria\Search\Search;
use IQDEV\ElasticSearch\Document\Property\Property;
use IQDEV\ElasticSearchTests\AbstractTestCase;
use IQDEV\ElasticSearchTests\Helpers\FormatData;
use IQDEV\ElasticSearchTests\Service\SearchClient;

class KeywordsTest extends AbstractTestCase
{
    /**
     * Поиск элементов по пустому фильтру свойства
     *
     * @return void
     */
    public function testExistByEmptyFilter(): void
    {
        $criteria = new Criteria();

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                FilterOperator::EQ,
                new FilterKeyword('')
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => []
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Поиск элементов по фильтру свойства
     *
     * @return void
     */
    public function testExistByFilter(): void
    {
        $criteria = new Criteria();

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                FilterOperator::EQ,
                new FilterKeyword('red')
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                's2',
                'h1'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    /**
     * Поиск элементов по фильтру свойства и категории
     *
     * @return void
     */
    public function testExistByFilterAndCategory(): void
    {
        $criteria = new Criteria();
        $criteria->getSearch()->add(
            new Search(
                new Property('category_id'),
                'shoes',
            ),
        );

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                FilterOperator::EQ,
                new FilterKeyword('red')
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                's2',
            ]
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Поиск элементов по нескольким свойствам
     *
     * @return void
     */
    public function testExistByMultipleFilter(): void
    {
        $criteria = new Criteria();

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                FilterOperator::EQ,
                new FilterKeyword('red')
            ),
            new Filter(
                new Field('size'),
                FilterOperator::EQ,
                new FilterKeyword('xxl')
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => []
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Поиск элементов по нескольким свойствам
     *
     * @return void
     */
    public function testExistByMultipleFilter2(): void
    {
        $criteria = new Criteria();

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                FilterOperator::EQ,
                new FilterKeyword('red')
            ),
            new Filter(
                new Field('size'),
                FilterOperator::EQ,
                new FilterKeyword('xl')
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'h1'
            ]
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Поиск элементов по нескольким свойствам и категории
     *
     * @return void
     */
    public function testExistByMultipleFilterAndCategory(): void
    {
        $criteria = new Criteria();

        $criteria->getSearch()->add(
            new Search(
                new Property('category_id'),
                'prices',
            ),
        );

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                FilterOperator::EQ,
                new FilterKeyword('red')
            ),
            new Filter(
                new Field('size'),
                FilterOperator::EQ,
                new FilterKeyword('xl')
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => []
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }
}