<?php

namespace IQDEV\ElasticSearchTests\Filter;

use IQDEV\ElasticSearchTests\AbstractTestCase;
use IQDEV\ElasticSearchTests\Helpers\FormatData;
use IQDEV\ElasticSearchTests\Service\SearchClient;
use IQDEV\Search\Criteria;
use IQDEV\Search\Document\Property\AttrType;
use IQDEV\Search\Document\Property\PropertyType;
use IQDEV\Search\Order\Order;
use IQDEV\Search\Order\OrderAscType;
use IQDEV\Search\Order\OrderDescType;
use IQDEV\Search\Query\SearchQuery;

class SortTest extends AbstractTestCase
{
    /**
     * Сортировка элементов по категории
     *
     * @return void
     */
    public function testSortByCategory(): void
    {
        $criteria = new Criteria();

        $criteria->sorting()->add(new Order(
            new AttrType('category_id'),
            new OrderAscType(),
        ));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'p1',
                's1',
                's2',
                's3',
                's4',
                'h1',
                'h2',
                'h3',
            ]
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Обратная сортировка элементов по категории
     *
     * @return void
     */
    public function testSortByCategoryReverse(): void
    {
        $criteria = new Criteria();

        $criteria->sorting()->add(new Order(
            new AttrType('category_id'),
            new OrderDescType(),
        ));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'h1',
                'h2',
                'h3',
                's1',
                's2',
                's3',
                's4',
                'p1',
            ]
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Сортировка элементов по свойству
     *
     * @return void
     */
    public function testSortByKeyword(): void
    {
        $criteria = new Criteria();

        $criteria->sorting()->add(new Order(
            new PropertyType('color', PropertyType::TYPE_KEYWORD),
            new OrderAscType(),
        ));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                's3',
                's1',
                's4',
                's2',
                'h1',
                'h2',
                'h3',
                'p1'
            ]
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Сортировка элементов по свойству
     *
     * @return void
     */
    public function testSortByKeywordReverse(): void
    {
        $criteria = new Criteria();

        $criteria->sorting()->add(new Order(
            new PropertyType('color', PropertyType::TYPE_KEYWORD),
            new OrderDescType(),
        ));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'h2',
                'h3',
                'p1',
                's2',
                'h1',
                's1',
                's4',
                's3',
            ]
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Сортировка элементов по свойству
     *
     * @return void
     */
    public function testSortByNumber(): void
    {
        $criteria = new Criteria();

        $criteria->sorting()->add(new Order(
            new PropertyType('price', PropertyType::TYPE_NUMBER),
            new OrderAscType(),
        ));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                's1',
                's2',
                's3',
                's4',
                'h1',
                'h2',
                'h3',
                'p1'
            ]
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Сортировка элементов по свойству
     *
     * @return void
     */
    public function testSortByNumberReverse(): void
    {
        $criteria = new Criteria();

        $criteria->sorting()->add(new Order(
            new PropertyType('price', PropertyType::TYPE_NUMBER),
            new OrderDescType(),
        ));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'p1',
                'h3',
                'h2',
                'h1',
                's4',
                's3',
                's2',
                's1',
            ]
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Сортировка элементов по свойству
     *
     * @return void
     */
    public function testSortByCombined(): void
    {
        $criteria = new Criteria();

        $criteria->sorting()->add(new Order(
            new PropertyType('size', PropertyType::TYPE_KEYWORD),
            new OrderAscType(),
        ));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                's4',
                's1',
                's2',
                's3',
                'h1',
                'h3',
                'p1',
                'h2',
            ]
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Сортировка элементов по свойству
     *
     * @return void
     */
    public function testSortByCombinedReverse(): void
    {
        $criteria = new Criteria();

        $criteria->sorting()->add(new Order(
            new PropertyType('size', PropertyType::TYPE_KEYWORD),
            new OrderDescType(),
        ));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'h2',
                'h1',
                'h3',
                'p1',
                's2',
                's3',
                's1',
                's4',
            ]
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }
}