<?php

namespace IQDEV\ElasticSearchTests\Filter;

use IQDEV\ElasticSearchTests\AbstractTestCase;
use IQDEV\ElasticSearchTests\Helpers\FormatData;
use IQDEV\ElasticSearchTests\Service\SearchClient;
use IQDEV\Search\Criteria;
use IQDEV\Search\Filter\Field;
use IQDEV\Search\Filter\Filter;
use IQDEV\Search\Filter\FilterGroupCollection;
use IQDEV\Search\Filter\FilterKeyword;
use IQDEV\Search\Filter\FilterOperator;
use IQDEV\Search\Filter\FilterType;
use IQDEV\Search\Query\SearchQuery;

class SearchItemsTest extends AbstractTestCase
{
    /**
     * Поиск всех элементов раздела
     *
     * @return void
     */
    public function testExistByCategory(): void
    {
        $criteria = new Criteria();

        $criteria->filters()->add(
            (new FilterGroupCollection([
                new Filter(
                    new Field('category_id'),
                    new FilterOperator(FilterOperator::EQ),
                    new FilterKeyword('prices')
                )
            ]))
                ->setFilterType(FilterType::query())
        );

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'p1'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    /**
     * Поиск всех элементов раздела
     *
     * @return void
     */
    public function testExistByEmptyCategory(): void
    {
        $criteria = new Criteria();

        $criteria->filters()->add(
            (new FilterGroupCollection([
                new Filter(
                    new Field('category_id'),
                    new FilterOperator(FilterOperator::EQ),
                    new FilterKeyword('')
                )
            ]))
                ->setFilterType(FilterType::query())
        );

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => []
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Поиск всех элементов
     *
     * @return void
     */
    public function testExistAllItems(): void
    {
        $criteria = new Criteria();

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                's1',
                's2',
                's3',
                's4',
                'h1',
                'h2',
                'h3',
                'p1'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    /**
     * Поиск существующего элемента по части строки поиска
     *
     * @return void
     */
    public function testExistItemsByPart(): void
    {
        $criteria = new Criteria();

        $criteria->filters()->add(
            (new FilterGroupCollection([
                new Filter(
                    new Field('search'),
                    new FilterOperator(FilterOperator::EQ),
                    new FilterKeyword('Nike')
                )
            ]))
                ->setFilterType(FilterType::query())
        );

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                's4',
                'h1',
                'h2',
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    /**
     * Поиск существующего элемента по полному названию
     *
     * @return void
     */
    public function testExistItemsByName(): void
    {
        $criteria = new Criteria();

        $criteria->filters()->add(
            (new FilterGroupCollection([
                new Filter(
                    new Field('search'),
                    new FilterOperator(FilterOperator::EQ),
                    new FilterKeyword('Nike Dri-FIT Strike')
                )
            ]))
                ->setFilterType(FilterType::query())
        );

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'h1',
                'h2',
                's4'
            ]
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Поиск существующего элемента по полному названию
     *
     * @return void
     */
    public function testExistItemsByCyrillicName(): void
    {
        $criteria = new Criteria();

        $criteria->filters()->add(
            (new FilterGroupCollection([
                new Filter(
                    new Field('search'),
                    new FilterOperator(FilterOperator::EQ),
                    new FilterKeyword('Товар с ценой')
                )
            ]))
                ->setFilterType(FilterType::query())
        );

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'p1',
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }
}