<?php

namespace IQDEV\ElasticSearchTests\Filter;

use IQDEV\ElasticSearchTests\AbstractTestCase;
use IQDEV\ElasticSearchTests\Helpers\FormatData;
use IQDEV\ElasticSearchTests\Service\SearchClient;
use IQDEV\Search\Criteria;
use IQDEV\Search\Filter\Field;
use IQDEV\Search\Filter\Filter;
use IQDEV\Search\Filter\FilterGroupCollection;
use IQDEV\Search\Filter\FilterKeyword;
use IQDEV\Search\Filter\FilterOperator;
use IQDEV\Search\Filter\FilterType;
use IQDEV\Search\Query\SearchQuery;

class KeywordsTest extends AbstractTestCase
{
    /**
     * Поиск элементов по пустому фильтру свойства
     *
     * @return void
     */
    public function testExistByEmptyFilter(): void
    {
        $criteria = new Criteria();

        $criteria->filters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('')
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => []
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Поиск элементов по фильтру свойства
     *
     * @return void
     */
    public function testExistByFilter(): void
    {
        $criteria = new Criteria();

        $criteria->filters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('red')
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                's2',
                'h1'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    /**
     * Поиск элементов по фильтру свойства и категории
     *
     * @return void
     */
    public function testExistByFilterAndCategory(): void
    {
        $criteria = new Criteria();

        $criteria->filters()->add(
            (new FilterGroupCollection([
                new Filter(
                    new Field('category_id'),
                    new FilterOperator(FilterOperator::EQ),
                    new FilterKeyword('shoes')
                )
            ]))
                ->setFilterType(FilterType::query())
        );

        $criteria->filters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('red')
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                's2',
            ]
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Поиск элементов по нескольким свойствам
     *
     * @return void
     */
    public function testExistByMultipleFilter(): void
    {
        $criteria = new Criteria();

        $criteria->filters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('red')
            ),
            new Filter(
                new Field('size'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('xxl')
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => []
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Поиск элементов по нескольким свойствам
     *
     * @return void
     */
    public function testExistByMultipleFilter2(): void
    {
        $criteria = new Criteria();

        $criteria->filters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('red')
            ),
            new Filter(
                new Field('size'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('xl')
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'h1'
            ]
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Поиск элементов по нескольким свойствам и категории
     *
     * @return void
     */
    public function testExistByMultipleFilterAndCategory(): void
    {
        $criteria = new Criteria();

        $criteria->filters()->add(
            (new FilterGroupCollection([
                new Filter(
                    new Field('category_id'),
                    new FilterOperator(FilterOperator::EQ),
                    new FilterKeyword('prices')
                )
            ]))
                ->setFilterType(FilterType::query())
        );

        $criteria->filters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('red')
            ),
            new Filter(
                new Field('size'),
                new FilterOperator(FilterOperator::EQ),
                new FilterKeyword('xl')
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => []
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }
}