<?php

namespace IQDEV\Search\SearchQuery;

use IQDEV\Search\Filter\FilterCategory;
use IQDEV\Search\Filter\FilterCollection;
use IQDEV\Search\Filter\FilterKeyword;
use IQDEV\Search\Filter\FilterNumber;

final class SearchQuery
{
    public ?FilterCollection $filters = null;
    public int $limit = 12;
    public int $page = 0;
    public string $sortBy = '';
    public string $sortDirection = 'asc';
    public ?string $sortType = null;
    public ?string $query = null;

    /**
     * @param array $data Массив параметров запроса:
     * Структура:
     * <ul>
     *  <li>
     *      filters
     *      <ul>
     *          <li>categoryId => 123</li>
     *          <li>['min' => null, 'max' => 123]</li>
     *          <li>красный | [красный, желтый]</li>
     *      </ul>
     *  </li>
     *  <li>
     *      pagination
     *      <ul>
     *          <li>limit => 123</li>
     *          <li>page => 1</li>
     *      </ul>
     *  </li>
     *  <li>
     *      query => 'красный'
     *  </li>
     *  <li>
     *      order
     *      <ul>
     *          <li>by => 'smth.'</li>
     *          <li>direction => 'asc | desc'</li>
     *      </ul>
     *  </li>
     * </ul>
     *
     * @return static
     */
    public static function createFromArray(array $data): self
    {
        $instance = new self();

        $filters = new FilterCollection();
        if (isset($data['filters'])) {
            foreach ($data['filters'] as $code => $value) {
                if ($code === 'categoryId') {
                    $filters->add(new FilterCategory($value));
                    continue;
                }

                if (is_array($value)) {
                    if (array_intersect(['min', 'max'], array_keys($value))) {
                        $value = array_map(static fn($v) => (float)$v, $value);
                        $filters->add(new FilterNumber($code, $value['min'] ?? 0, $value['max'] ?? PHP_INT_MAX));
                        continue;
                    }

                    $value = array_map(static fn($v) => (string)$v, $value);
                } else {
                    $value = (string) $value;
                }

                $filters->add(new FilterKeyword($code, $value));
            }
        }

        if ($filters->isEmpty() === false) {
            $instance->filters = $filters;
        }

        if (isset($data['pagination']['limit'])) {
            $instance->limit = (int) $data['pagination']['limit'];
        }

        if (isset($data['pagination']['page'])) {
            $instance->page = (int) $data['pagination']['page'];
        }

        if (isset($data['query'])) {
            $instance->query = $data['query'];
        }

        if (isset($data['order'])) {
            if (isset($data['order']['by'])) {
                $instance->sortBy = $data['order']['by'];
            }

            if (isset($data['order']['direction'])) {
                $instance->sortDirection = $data['order']['direction'];
            }
            
            if (isset($data['order']['type'])) {
                // todo временное
                $instance->sortType = $data['order']['type'];
            }
        }

        return $instance;
    }
}
