<?php

declare(strict_types=1);

namespace IQDEV\ElasticSearchTests\Filter;

use IQDEV\ElasticSearch\Criteria\Criteria;
use IQDEV\ElasticSearch\Criteria\Filter\Collection\FilterGroupCollection;
use IQDEV\ElasticSearch\Criteria\Filter\Field;
use IQDEV\ElasticSearch\Criteria\Filter\Filter;
use IQDEV\ElasticSearch\Criteria\Filter\FilterOperator;
use IQDEV\ElasticSearch\Criteria\Filter\LogicOperator;
use IQDEV\ElasticSearch\Criteria\Filter\Value\FilterKeyword;
use IQDEV\ElasticSearch\Criteria\Query\SearchQuery;
use IQDEV\ElasticSearch\Criteria\Search\Search;
use IQDEV\ElasticSearch\Document\Property\Property;
use IQDEV\ElasticSearchTests\AbstractTestCase;
use IQDEV\ElasticSearchTests\Helpers\FormatData;
use IQDEV\ElasticSearchTests\Service\SearchClient;

class UpdatedSearchTest extends AbstractTestCase
{
    /**
     * Поиск по свойству nested
     *
     * @return void
     */
    public function testSearchByNestedProperty(): void
    {
        $criteria = new Criteria();

        $criteria->getSearch()->add(
            new Search(
                new Property('brand'),
                'rebook',
            ),
        );

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                's3',
                'h3',
                'p1'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    /**
     * Поиск по nested полю и обычному свойству одноврмененно
     *
     * @return void
     */
    public function testSearchByNestedAndNonNestedProperty(): void
    {
        $criteria = new Criteria();

        $criteria->getSearch()->add(
            new Search(
                new Property('brand'),
                'rebook',
            ),
        );

        $criteria->getSearch()->add(
            new Search(
                new Property('category_id'),
                'prices',
            ),
        );

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                'p1'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    /**
     * Поиск по нескольким nested полям
     *
     * @return void
     */
    public function testSearchByDifferentNestedProperty(): void
    {
        $criteria = new Criteria();

        $criteria->getSearch()->add(
            new Search(
                new Property('brand'),
                'rebook',
            ),
        );

        $criteria->getSearch()->add(
            new Search(
                new Property('color'),
                'white',
            ),
        );

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                'h3',
                'p1'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    public function testQueryNonNested(): void
    {
        $criteria = new Criteria();

        $filterCollection = new FilterGroupCollection([
            new Filter(
                new Field('category_id'),
                FilterOperator::CONTAINS,
                new FilterKeyword('shoes'),
            )
        ]);
        $criteria->getFilters()->add($filterCollection);

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                's1',
                's2',
                's3',
                's4',
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    public function testQueryNonNestedMoreLogicOr(): void
    {
        $criteria = new Criteria();

        $filterCollection = new FilterGroupCollection([
            new Filter(
                new Field('category_id'),
                FilterOperator::CONTAINS,
                new FilterKeyword('shoes'),
            ),
            new Filter(
                new Field('category_id'),
                FilterOperator::CONTAINS,
                new FilterKeyword('t-short'),
            ),
        ]);
        $filterCollection->setLogicOperator(LogicOperator::OR);
        $criteria->getFilters()->add($filterCollection);

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                's1',
                's2',
                's3',
                's4',
                'h1',
                'h2',
                'h3',
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    public function testQueryNonNestedMoreLogicAnd(): void
    {
        $criteria = new Criteria();

        $filterCollection = new FilterGroupCollection([
            new Filter(
                new Field('category_id'),
                FilterOperator::CONTAINS,
                new FilterKeyword('shoes'),
            ),
            new Filter(
                new Field('category_id'),
                FilterOperator::CONTAINS,
                new FilterKeyword('t-short'),
            ),
        ]);
        $criteria->getFilters()->add($filterCollection);

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    public function testMustNotNestedFilter(): void
    {
        $criteria = new Criteria();
        $group = new FilterGroupCollection([
            new Filter(
                new Field('brand'),
                FilterOperator::EQ,
                new FilterKeyword('adidas'),
            )
        ]);
        $group->setLogicOperator(LogicOperator::NOT);

        $criteria->getFilters()->add($group);

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                's3',
                's4',
                'h1',
                'h2',
                'h3',
                'p1',
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    public function testMustNotPropertyFilter(): void
    {
        $criteria = new Criteria();
        $group = new FilterGroupCollection([
            new Filter(
                new Field('category_id'),
                FilterOperator::EQ,
                new FilterKeyword('prices'),
            )
        ]);
        $group->setLogicOperator(LogicOperator::NOT);

        $criteria->getFilters()->add($group);

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                's1',
                's2',
                's3',
                's4',
                'h1',
                'h2',
                'h3',
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }
}
