<?php

namespace IQDEV\ElasticSearch\Document;

use IQDEV\ElasticSearch\Config\MappingValidator;
use IQDEV\ElasticSearch\Configuration;
use IQDEV\ElasticSearch\Facet\Collection\FacetCollection;
use IQDEV\ElasticSearch\Facet\Facet;
use IQDEV\ElasticSearch\Helper\ArrayHelper;

class ProductDocument implements Document
{
    private array $properties = [];
    private FacetCollection $keywordFacets;
    private FacetCollection $numberFacets;
    private ?string $searchContent = null;
    private array $info = [];
    private bool $skipEmpty = false;

    public function __construct(
        private Facet $categoryFacet
    ) {
        $this->keywordFacets = new FacetCollection();
        $this->numberFacets = new FacetCollection();
    }

    public static function create(Facet $categoryFacet): self
    {
        return new self($categoryFacet);
    }

    /**
     * @return FacetCollection
     */
    public function getKeywordFacets(): FacetCollection
    {
        return $this->keywordFacets;
    }

    /**
     * @return Facet
     */
    public function getCategoryFacet(): Facet
    {
        return $this->categoryFacet;
    }

    /**
     * @return FacetCollection
     */
    public function getNumberFacets(): FacetCollection
    {
        return $this->numberFacets;
    }

    /**
     * @param string|null $searchContent
     */
    public function setSearchContent(?string $searchContent): void
    {
        $this->searchContent = $searchContent;
    }

    public function setAdditionData(array $info): self
    {
        $this->info = $info;
        return $this;
    }

    /**
     * Установка значения свойства документа индекса по параметрам конфигурации.
     * Имеет приоритет по сравнению с вызовами функций для установки данных.
     *
     * @param Configuration $configuration
     * @param string $property
     * @param $value
     *
     * @return $this
     */
    public function setByConfiguration(Configuration $configuration, string $property, $value): self
    {
        if (!MappingValidator::isPropertyExists($configuration, $property)) {
            throw new \InvalidArgumentException('Property ' . $property . ' doesnt exist');
        }

        $this->properties[$property] = $value;

        return $this;
    }

    public function skipEmpty(bool $skipEmpty = false): self
    {
        $this->skipEmpty = $skipEmpty;

        return $this;
    }

    public function es(): array
    {
        $document = [
            'category_id' => $this->getCategoryFacet()->es()['category_id'],
            'search_data' => [
                'keyword_facet' => $this->getKeywordFacets()->es(),
                'number_facet' => $this->getNumberFacets()->es()
            ],
            'data' => $this->info
        ];

        if (isset($this->searchContent)) {
            $document['full_search_content'] = $this->searchContent;
            $document['suggest_search_content'] = $this->searchContent;
        }

        $result = array_replace_recursive($document, $this->properties);
        if (true === $this->skipEmpty) {
            $result = ArrayHelper::array_filter_recursive($result, static fn ($val) => $val !== null || $val === false);
        }

        return $result;
    }
}
