<?php

namespace IQDEV\ElasticSearchTests\Filter;

use IQDEV\ElasticSearch\Criteria\Criteria;
use IQDEV\ElasticSearch\Criteria\Query\SearchQuery;
use IQDEV\ElasticSearch\Criteria\Search\Search;
use IQDEV\ElasticSearch\Document\Property\Property;
use IQDEV\ElasticSearch\Document\Property\PropertyType;
use IQDEV\ElasticSearchTests\AbstractTestCase;
use IQDEV\ElasticSearchTests\Helpers\FormatData;
use IQDEV\ElasticSearchTests\Service\SearchClient;

class SearchItemsTest extends AbstractTestCase
{
    /**
     * Поиск всех элементов раздела
     *
     * @return void
     */
    public function testExistByCategory(): void
    {
        $criteria = new Criteria();
        $criteria->getSearch()->add(
            new Search(
                new Property('category_id'),
                'prices'
            )
        );

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'p1'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    /**
     * Поиск всех элементов раздела
     *
     * @return void
     */
    public function testExistByEmptyCategory(): void
    {
        $criteria = new Criteria();

        $criteria->getSearch()->add(
            new Search(
                new Property('category_id'),
                '',
            )
        );

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => []
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Поиск всех элементов
     *
     * @return void
     */
    public function testExistAllItems(): void
    {
        $criteria = new Criteria();

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                's1',
                's2',
                's3',
                's4',
                'h1',
                'h2',
                'h3',
                'p1'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    /**
     * Поиск существующего элемента по части строки поиска
     *
     * @return void
     */
    public function testExistItemsByPart(): void
    {
        $criteria = new Criteria();

        $criteria->getSearch()->add(
            new Search(
                new Property('full_search_content', PropertyType::TEXT),
                'Nike'
            )
        );

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                's4',
                'h1',
                'h2',
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    /**
     * Поиск существующего элемента по полному названию
     *
     * @return void
     */
    public function testExistItemsByName(): void
    {
        $criteria = new Criteria();

        $criteria->getSearch()->add(
            new Search(
                new Property('full_search_content', PropertyType::TEXT),
                'Nike Dri-FIT Strike',
            ),
        );

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'h1',
                'h2',
                's4'
            ]
        ];

        $this->assertEquals($expected, FormatData::formatData($result));
    }

    /**
     * Поиск существующего элемента по полному названию
     *
     * @return void
     */
    public function testExistItemsByCyrillicName(): void
    {
        $criteria = new Criteria();

        $criteria->getSearch()->add(
            new Search(
                new Property('full_search_content', PropertyType::TEXT),
                'Товар с ценой',
            ),
        );

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                'p1',
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }
}
