<?php

namespace IQDEV\ElasticSearchTests\Filter;

use IQDEV\ElasticSearch\Criteria\Criteria;
use IQDEV\ElasticSearch\Criteria\Filter\Collection\FilterGroupCollection;
use IQDEV\ElasticSearch\Criteria\Filter\Field;
use IQDEV\ElasticSearch\Criteria\Filter\Filter;
use IQDEV\ElasticSearch\Criteria\Filter\FilterOperator;
use IQDEV\ElasticSearch\Criteria\Filter\Value\FilterKeyword;
use IQDEV\ElasticSearch\Criteria\Filter\Value\FilterNumber;
use IQDEV\ElasticSearch\Criteria\Query\SearchQuery;
use IQDEV\ElasticSearchTests\AbstractTestCase;
use IQDEV\ElasticSearchTests\Helpers\FormatData;
use IQDEV\ElasticSearchTests\Service\SearchClient;

class CommonRangeKeywordsTest extends AbstractTestCase
{
    /**
     * Поиск элементов по фильтру свойства и цене
     *
     * @return void
     */
    public function testExistByFilterAndMinPrice(): void
    {
        $criteria = new Criteria();

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                FilterOperator::EQ,
                new FilterKeyword('red')
            )
        ]));

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('price'),
                FilterOperator::GT,
                new FilterNumber(102)
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                'h1'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    /**
     * Поиск элементов по фильтру свойства и цене
     *
     * @return void
     */
    public function testExistByFilterAndMaxPrice(): void
    {
        $criteria = new Criteria();

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                FilterOperator::EQ,
                new FilterKeyword('red')
            )
        ]));

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('price'),
                FilterOperator::LT,
                new FilterNumber(102)
            )
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;


        $expected = [
            'hits' => [
                's2'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }

    /**
     * Поиск элементов по фильтру свойства и цене
     *
     * @return void
     */
    public function testExistByFilterAndRangePrice(): void
    {
        $criteria = new Criteria();

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('color'),
                FilterOperator::EQ,
                new FilterKeyword('red')
            )
        ]));

        $criteria->getFilters()->add(new FilterGroupCollection([
            new Filter(
                new Field('price'),
                FilterOperator::GTE,
                new FilterNumber(101)
            ),
            new Filter(
                new Field('price'),
                FilterOperator::LTE,
                new FilterNumber(104)
            ),
        ]));

        $q = new SearchQuery($criteria);

        $handler = SearchClient::getInstance();
        $result = $handler->handle($q)->result;

        $expected = [
            'hits' => [
                'h1',
                's2'
            ]
        ];

        $this->assertEqualsCanonicalizing($expected, FormatData::formatData($result));
    }
}