<?php

declare(strict_types=1);

namespace IQDEV\Tests\Packages\DoctrineHttpFilter;

use Doctrine\ORM\AbstractQuery;
use IQDEV\Packages\DoctrineHttpFilter\Filter\Like;
use IQDEV\Packages\DoctrineHttpFilter\HttpFilter;
use IQDEV\Packages\DoctrineHttpFilter\HttpSort;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Entity\Comment;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Entity\Post;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Repository\PostRepository;
use Symfony\Component\HttpFoundation\Request;

class HttpSortTest extends TestCase
{
    public function testNumberSortWithFilter(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $title = 'Видеопредставление';
        $title2 = 'Видео';
        $countOfViews = 12;
        $countOfViews2 = 22;

        $post = new Post(
            $title,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $countOfViews
        );

        $post2 = new Post(
            $title2,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $countOfViews2
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'title' => Like::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => $title2,
            ],
        ]));

        $result = $postRepository->createQueryForSort(
            new Request([
                HttpSort::REQUEST_SORT_KEY => [
                    'countOfViews' => 'DESC',
                ],
            ]),
            $result
        )
            ->getQuery()
            ->getResult();

        $this->assertSame($countOfViews, $result[1]->countOfViews);
        $this->assertCount(2, $result);
    }

    public function testStringSortWithFilter(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $title = 'Видеопредставление';
        $title2 = 'Дорога';

        $post = new Post(
            $title,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $post2 = new Post(
            $title2,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'title' => Like::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => $title2,
            ],
        ]));

        $result = $postRepository->createQueryForSort(
            new Request([
                HttpSort::REQUEST_SORT_KEY => [
                    'title' => 'ASC',
                ],
            ]),
            $result
        )
            ->getQuery()
            ->getResult();

        $this->assertSame($title2, $result[0]->title);
        $this->assertCount(1, $result);
    }

    public function testDateSortWithFilter(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $title = 'Видеопредставление';
        $title2 = 'Видео';
        $date = \DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-01 12:00:00');
        $date2 = \DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-05 19:30:00');

        $post = new Post(
            $title,
            $this->faker->text(),
            $this->faker->boolean(),
            $date
        );

        $post2 = new Post(
            $title2,
            $this->faker->text(),
            $this->faker->boolean(),
            $date2
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'title' => Like::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => $title2,
            ],
        ]));

        $result = $postRepository->createQueryForSort(
            new Request([
                HttpSort::REQUEST_SORT_KEY => [
                    'createdAt' => 'DESC',
                ],
            ]),
            $result
        )
            ->getQuery()
            ->getResult();

        $this->assertSame($date2, $result[0]->createdAt);
        $this->assertCount(2, $result);
    }

    public function testSortWithJoinTable(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $author1 = 'Александр';
        $author2 = 'Яков';

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $comment1 = new Comment(
            $author1,
            $this->faker->text(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            $post
        );
        $comment2 = new Comment(
            $author1,
            $this->faker->text(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            $post
        );
        $comment3 = new Comment(
            $author2,
            $this->faker->text(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            $post
        );

        $this->em->persist($post);
        $this->em->persist($comment1);
        $this->em->persist($comment2);
        $this->em->persist($comment3);
        $this->em->flush();

        $result = $postRepository->createQueryForSort(
            new Request([
                HttpSort::REQUEST_SORT_KEY => [
                    'c.author' => 'DESC',
                ],
            ])
        )
            ->leftJoin(Comment::class, 'c')
            ->select('p', 'c.author')
            ->getQuery()
            ->getResult(AbstractQuery::HYDRATE_ARRAY);

        $this->assertSame($author2, $result[0]['author']);
    }
}
