<?php

declare(strict_types=1);

namespace IQDEV\Tests\Packages\DoctrineHttpFilter;

use DateTimeImmutable;
use DateInterval;
use IQDEV\Packages\DoctrineHttpFilter\Filter\Date;
use IQDEV\Packages\DoctrineHttpFilter\Filter\DateRange;
use IQDEV\Packages\DoctrineHttpFilter\Filter\In;
use IQDEV\Packages\DoctrineHttpFilter\Filter\Like;
use IQDEV\Packages\DoctrineHttpFilter\Filter\Range;
use IQDEV\Packages\DoctrineHttpFilter\Filter\Where;
use IQDEV\Packages\DoctrineHttpFilter\HttpFilter;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Entity\Post;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Repository\PostRepository;
use Symfony\Component\HttpFoundation\Request;

class FilterByMultipleFiltersTest extends TestCase
{
    public function testFilterByMultipleFiltersLikeAndDateRange(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstName = 'Видеопредставление';
        $firstDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-01 12:00:00');
        $secondName = 'Видеомагнитофон';
        $secondDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-05 14:00:00');

        $post = new Post(
            $firstName,
            $this->faker->text(),
            $this->faker->boolean(),
            $firstDate,
        );

        $post2 = new Post(
            $secondName,
            $this->faker->text(),
            $this->faker->boolean(),
            $secondDate,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $subtitle = 'Видео';
        $firstDateWithDay = $firstDate->add(new DateInterval('P1D'));
        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
            'title' => Like::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => $subtitle,
                'createdAt' => [
                    'from' => $firstDate->format('Y-m-d'),
                    'to' => $firstDateWithDay->format('Y-m-d'),
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(1, $result);
    }

    public function testFilterByMultipleFiltersDateAndIn(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstName = 'Спорт';
        $firstDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-01 12:00:00');
        $secondName = 'Наука';
        $secondDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-01 14:00:00');

        $post = new Post(
            $firstName,
            $this->faker->text(),
            $this->faker->boolean(),
            $firstDate,
        );

        $post2 = new Post(
            $secondName,
            $this->faker->text(),
            $this->faker->boolean(),
            $secondDate,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $inValues = ['Спорт', 'Бизнес', 'Кулинария'];
        $result = $postRepository->createQueryByFilter([
            'createdAt' => Date::class,
            'title' => In::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => $inValues,
                'createdAt' => $firstDate->format('Y-m-d'),
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(1, $result);
    }

    public function testFilterByMultipleFiltersRangeAndWhere(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstCountOfViews = 20;
        $secondCountOfViews = 40;

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $firstCountOfViews,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $secondCountOfViews,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $minValue = 30;
        $name = 'Not found';
        $result = $postRepository->createQueryByFilter([
            'title' => Where::class,
            'countOfViews' => Range::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => $name,
                'createdAt' => [
                    'min' => $minValue,
                ]
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertEmpty($result);
    }
}
