<?php

declare(strict_types=1);

namespace IQDEV\Tests\Packages\DoctrineHttpFilter;

use DateInterval;
use DateTimeImmutable;
use IQDEV\Packages\DoctrineHttpFilter\Filter\Date;
use IQDEV\Packages\DoctrineHttpFilter\HttpFilter;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Entity\Post;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Repository\PostRepository;
use Symfony\Component\HttpFoundation\Request;

class FilterByDateTest extends TestCase
{
    public function testSuccessFilterDateWithOneResult(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $date = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-01 12:00:00');

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $date,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => Date::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => $date->format('Y-m-d'),
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        foreach ($result as $post) {
            $this->assertEquals($date->format('Y-m-d'), $post->createdAt->format('Y-m-d'));
        }
    }

    public function testSuccessFilterDateWithoutTime(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $date = DateTimeImmutable::createFromFormat('Y-m-d', '2020-01-01');

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $date,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => Date::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => $date->format('Y-m-d'),
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        foreach ($result as $post) {
            $this->assertEquals($date->format('Y-m-d'), $post->createdAt->format('Y-m-d'));
        }
    }

    public function testSuccessFilterDateWithSeveralResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-01 12:00:00');
        $secondDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-01 19:30:00');

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $firstDate,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $secondDate,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => Date::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => $firstDate->format('Y-m-d'),
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        foreach ($result as $post) {
            $this->assertEquals($firstDate->format('Y-m-d'), $post->createdAt->format('Y-m-d'));
        }
    }

    public function testSuccessFilterDateRangeWithoutResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-01 14:00:00');

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $firstDate,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => Date::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => $firstDate->add(new DateInterval('P100Y'))->format('Y-m-d'),
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertEmpty($result);
    }

    public function testFilterDateWithoutParameterValue(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $countPosts = $postRepository->count();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => Date::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertCount($countPosts, $result);
    }

    public function testFilterDateWithNotDateParameterValue(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $countPosts = $postRepository->count();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => Date::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => 'not date',
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertCount($countPosts, $result);
    }

    public function testFilterDateWithDifferentKey(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $countPosts = $postRepository->count();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => Date::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'notCreatedAt' => 'Not createdAt value',
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertCount($countPosts, $result);
    }
}
