<?php

declare(strict_types=1);

namespace IQDEV\Tests\Packages\DoctrineHttpFilter;

use DateInterval;
use DateTimeImmutable;
use IQDEV\Packages\DoctrineHttpFilter\Filter\DateRange;
use IQDEV\Packages\DoctrineHttpFilter\HttpFilter;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Entity\Post;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Repository\PostRepository;
use Symfony\Component\HttpFoundation\Request;

class FilterByDateRangeTest extends TestCase
{
    public function testSuccessFilterDateRangeWithOneResult(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-01 12:00:00');
        $secondDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-05 14:00:00');

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $firstDate,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $secondDate,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $firstDateWithDay = $firstDate->add(new DateInterval('P1D'));
        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => [
                    'from' => $firstDate->format('Y-m-d'),
                    'to' => $firstDateWithDay->format('Y-m-d'),
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        foreach ($result as $post) {
            $this->assertGreaterThanOrEqual($firstDate->setTime(0, 0), $post->createdAt);
            $this->assertLessThanOrEqual($firstDateWithDay->setTime(23, 59, 59), $post->createdAt);
        }
    }

    public function testSuccessFilterDateRangeWithSeveralResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-01 12:00:00');
        $secondDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-05 11:00:00');

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $firstDate,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $secondDate,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $firstDateWithDays = $firstDate->add(new DateInterval('P10D'));
        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => [
                    'from' => $firstDate->format('Y-m-d'),
                    'to' => $firstDateWithDays->format('Y-m-d'),
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        foreach ($result as $post) {
            $this->assertGreaterThanOrEqual($firstDate->setTime(0, 0), $post->createdAt);
            $this->assertLessThanOrEqual($firstDateWithDays->setTime(23, 59, 59), $post->createdAt);
        }
    }

    public function testSuccessFilterDateRangeWithBoundaryResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-01 13:30:00');
        $secondDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-05 12:00:00');

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $firstDate,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $secondDate,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => [
                    'from' => $firstDate->format('Y-m-d'),
                    'to' => $secondDate->format('Y-m-d'),
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        foreach ($result as $post) {
            $this->assertGreaterThanOrEqual($firstDate->setTime(0, 0), $post->createdAt);
            $this->assertLessThanOrEqual($secondDate->setTime(23, 59, 59), $post->createdAt);
        }
    }

    public function testSuccessFilterDateRangeWithoutResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-01 15:00:00');
        $secondDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-05 00:00:00');

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $firstDate,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $secondDate,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => [
                    'from' => $firstDate->add(new DateInterval('P100Y'))->format('Y-m-d'),
                    'to' => $secondDate->add(new DateInterval('P100Y'))->format('Y-m-d'),
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertEmpty($result);
    }

    public function testFilterDateRangeWithoutParameterValue(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $countPosts = $postRepository->count();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertCount($countPosts, $result);
    }

    public function testFilterDateRangeWithNotDateParameterValue(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $countPosts = $postRepository->count();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => 'not date',
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertCount($countPosts, $result);
    }

    public function testFilterDateRangeWithNotDateParameterValues(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $countPosts = $postRepository->count();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => [
                    'from' => 'not date',
                    'to' => 'not date',
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertCount($countPosts, $result);
    }

    public function testFilterDateRangeWithDifferentKey(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $countPosts = $postRepository->count();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'notCreatedAt' => 'Not createdAt value',
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertCount($countPosts, $result);
    }

    public function testFilterDateRangeWithOnlyFromParameterValue(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-01 13:30:00');
        $secondDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-05 12:00:00');

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $firstDate,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $secondDate,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => [
                    'from' => $secondDate->format('Y-m-d'),
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertGreaterThanOrEqual(1, $result);
    }

    public function testFilterDateRangeWithOnlyToParameterValue(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-01 13:30:00');
        $secondDate = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', '2020-01-05 12:00:00');

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $firstDate,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $secondDate,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => [
                    'to' => $firstDate->format('Y-m-d'),
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertGreaterThanOrEqual(1, $result);
    }
}
