<?php

declare(strict_types=1);

namespace IQDEV\Tests\Packages\DoctrineHttpFilter;

use Doctrine\Common\DataFixtures\Executor\ORMExecutor;
use Doctrine\Common\DataFixtures\Loader;
use Doctrine\Common\DataFixtures\Purger\ORMPurger;
use Doctrine\DBAL\DriverManager;
use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\ORMSetup;
use Doctrine\ORM\Tools\SchemaTool;
use DoctrineExtensions\Query\Sqlite\StrfTime;
use Faker\Factory;
use Faker\Generator;
use IQDEV\Packages\DoctrineHttpFilter\HttpFilterEntityRepository;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Fixture\BaseFixture;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Fixture\CommentFixture;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Fixture\PostFixture;
use PHPUnit\Framework\TestCase as BaseTestCase;
use Symfony\Component\HttpFoundation\Request;

abstract class TestCase extends BaseTestCase
{
    protected EntityManagerInterface $em;
    protected Generator $faker;

    protected function setUp(): void
    {
        $this->em = $this->makeEntityManager();

        $schemaTool = new SchemaTool($this->em);
        $schemaTool->createSchema($this->em->getMetadataFactory()->getAllMetadata());

        $this->faker = Factory::create();

        $this->loadFixtures([
            new PostFixture(count: 20),
            new CommentFixture(200),
        ]);

        $this->em->getConnection()->beginTransaction();
    }

    protected function tearDown(): void
    {

        if ($this->em->isOpen()) {
            $this->em->getConnection()->rollBack();
            $this->em->close();
        }
        unset($this->em);
    }

    protected function makeEntityManager(): EntityManagerInterface
    {
        $config = ORMSetup::createAttributeMetadataConfiguration(
            [__DIR__ . '/Entity'],
            true,
        );
        $config->addCustomStringFunction('strftime', StrfTime::class);

        $connection = DriverManager::getConnection([
            'driver' => 'pdo_sqlite',
            'memory' => true,
        ], $config);

        return new EntityManager(
            $connection,
            $config,
        );
    }

    protected function makeRepositoryWithRequest(string $repositoryClass, Request $request): HttpFilterEntityRepository
    {
        /** @var HttpFilterEntityRepository $repoMock */
        $repoMock = $this->getMockBuilder($repositoryClass)
            ->setConstructorArgs([$this->em])
            ->onlyMethods(['getRequest'])
            ->getMock();

        $repoMock
            ->method('getRequest')
            ->willReturn($request);

        return $repoMock;
    }

    /** @param iterable<BaseFixture> $fixtures */
    protected function loadFixtures(iterable $fixtures): void
    {
        $fixtureLoader = new Loader();

        /** @var BaseFixture $fixture */
        foreach ($fixtures as $fixture) {
            $fixtureLoader->addFixture($fixture);
        }

        $purger = new ORMPurger();
        $executor = new ORMExecutor($this->em, $purger);
        $executor->execute($fixtureLoader->getFixtures());
    }
}
