<?php

declare(strict_types=1);

namespace IQDEV\Tests\Packages\DoctrineHttpFilter;

use DateInterval;
use DateTimeImmutable;
use IQDEV\Packages\DoctrineHttpFilter\Filter\DateRange;
use IQDEV\Packages\DoctrineHttpFilter\HttpFilter;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Entity\Post;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Repository\PostRepository;
use Symfony\Component\HttpFoundation\Request;

class FilterByDateRangeTest extends TestCase
{
    public function testSuccessFilterDateRangeWithOneResult(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstDate = DateTimeImmutable::createFromFormat('Y-m-d', '2020-01-01');
        $secondDate = DateTimeImmutable::createFromFormat('Y-m-d', '2020-01-05');

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $firstDate,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $secondDate,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => [
                    'from' => $firstDate->format('Y-m-d'),
                    'to' => $firstDate->add(new DateInterval('P1D'))->format('Y-m-d'),
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(1, $result);
    }

    public function testSuccessFilterDateRangeWithSeveralResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstDate = DateTimeImmutable::createFromFormat('Y-m-d', '2020-01-01');
        $secondDate = DateTimeImmutable::createFromFormat('Y-m-d', '2020-01-05');

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $firstDate,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $secondDate,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => [
                    'from' => $firstDate->format('Y-m-d'),
                    'to' => $firstDate->add(new DateInterval('P10D'))->format('Y-m-d'),
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(2, $result);
    }

    public function testSuccessFilterDateRangeWithBoundaryResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstDate = DateTimeImmutable::createFromFormat('Y-m-d', '2020-01-01');
        $secondDate = DateTimeImmutable::createFromFormat('Y-m-d', '2020-01-05');

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $firstDate,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $secondDate,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => [
                    'from' => $firstDate->format('Y-m-d'),
                    'to' => $secondDate->format('Y-m-d'),
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(2, $result);
    }

    public function testSuccessFilterDateRangeWithoutResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstDate = DateTimeImmutable::createFromFormat('Y-m-d', '2020-01-01');
        $secondDate = DateTimeImmutable::createFromFormat('Y-m-d', '2020-01-05');

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $firstDate,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            $secondDate,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'createdAt' => DateRange::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'createdAt' => [
                    'from' => $firstDate->add(new DateInterval('P1Y'))->format('Y-m-d'),
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertEmpty($result);
    }
}
