<?php

declare(strict_types=1);

namespace IQDEV\Tests\Packages\DoctrineHttpFilter;

use IQDEV\Packages\DoctrineHttpFilter\Exception\FilterParameterValueIsNullException;
use IQDEV\Packages\DoctrineHttpFilter\Filter\Where;
use IQDEV\Packages\DoctrineHttpFilter\HttpFilter;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Entity\Post;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Repository\PostRepository;
use Symfony\Component\HttpFoundation\Request;

class FilterByWhereTest extends TestCase
{
    public function testSuccessFilterWhereWithOneResult(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $title = $this->faker->name();

        $post = new Post(
            $title,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'title' => Where::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => $title,
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(1, $result);
        $this->assertEquals($title, current($result)->title);
    }

    public function testSuccessFilterWhereWithSeveralResult(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $title = $this->faker->name();

        $post = new Post(
            $title,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $post2 = new Post(
            $title,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'title' => Where::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => $title,
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(2, $result);
        $this->assertEquals($title, current($result)->title);
        $this->assertEquals($title, next($result)->title);
    }

    public function testFilterWhereWithNotResult(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $result = $postRepository->createQueryByFilter([
            'title' => Where::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => 'Стимул',
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertEmpty($result);
    }

    public function testFilterWhereWithoutParameterValue(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $this->expectException(FilterParameterValueIsNullException::class);

        $postRepository->createQueryByFilter([
            'title' => Where::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [],
        ]))
            ->getQuery()
            ->getResult();
    }

    public function testFilterWhereWithDifferentKey(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $this->expectException(FilterParameterValueIsNullException::class);

        $postRepository->createQueryByFilter([
            'title' => Where::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'notTitle' => 'Not title value',
            ],
        ]))
            ->getQuery()
            ->getResult();
    }
}
