<?php

declare(strict_types=1);

namespace IQDEV\Tests\Packages\DoctrineHttpFilter;

use DateTimeImmutable;
use IQDEV\Packages\DoctrineHttpFilter\Exception\FilterParameterValueIsNullException;
use IQDEV\Packages\DoctrineHttpFilter\Filter\Range;
use IQDEV\Packages\DoctrineHttpFilter\HttpFilter;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Entity\Post;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Repository\PostRepository;
use Symfony\Component\HttpFoundation\Request;

class FilterByRangeTest extends TestCase
{
    public function testSuccessFilterRangeWithOneResult(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $countOfViews = 20;

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $countOfViews,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'countOfViews' => Range::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'countOfViews' => [
                    'min' => 0,
                    'max' => 40,
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(1, $result);
    }

    public function testSuccessFilterRangeWithSeveralResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstCountOfViews = 20;
        $secondCountOfViews = 35;

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $firstCountOfViews,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $secondCountOfViews,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'countOfViews' => Range::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'countOfViews' => [
                    'min' => 10,
                    'max' => 40,
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(2, $result);
    }

    public function testSuccessFilterRangeWithBoundaryResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstCountOfViews = 10;
        $secondCountOfViews = 15;

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $firstCountOfViews,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $secondCountOfViews,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'countOfViews' => Range::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'countOfViews' => [
                    'min' => $firstCountOfViews,
                    'max' => $secondCountOfViews,
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(2, $result);
    }

    public function testSuccessFilterRangeWithoutResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstCountOfViews = 25;
        $secondCountOfViews = 35;

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $firstCountOfViews,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $secondCountOfViews,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'countOfViews' => Range::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'countOfViews' => [
                    'min' => 100,
                    'max' => 200,
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertEmpty($result);
    }

    public function testFilterRangeWithoutParameterValue(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $this->expectException(FilterParameterValueIsNullException::class);

        $postRepository->createQueryByFilter([
            'createdAt' => Range::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [],
        ]))
            ->getQuery()
            ->getResult();
    }

    public function testFilterRangeWithNotNumberParameterValue(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $this->expectException(FilterParameterValueIsNullException::class);

        $postRepository->createQueryByFilter([
            'countOfViews' => Range::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'countOfViews' => 'not number',
            ],
        ]))
            ->getQuery()
            ->getResult();
    }

    public function testFilterRangeWithNotNumberParameterValues(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $this->expectException(FilterParameterValueIsNullException::class);

        $postRepository->createQueryByFilter([
            'countOfViews' => Range::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'countOfViews' => [
                    'min' => 'not number',
                    'max' => 'not number',
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();
    }

    public function testFilterRangeWithDifferentKey(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $this->expectException(FilterParameterValueIsNullException::class);

        $postRepository->createQueryByFilter([
            'countOfViews' => Range::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'notCountOfViews' => 'Not countOfViews value',
            ],
        ]))
            ->getQuery()
            ->getResult();
    }

    public function testFilterRangeWithOnlyMinParameterValue(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstCountOfViews = 25;
        $secondCountOfViews = 35;

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $firstCountOfViews,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $secondCountOfViews,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'countOfViews' => Range::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'countOfViews' => [
                    'min' => 20,
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(2, $result);
    }

    public function testFilterRangeWithOnlyMaxParameterValue(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $firstCountOfViews = 25;
        $secondCountOfViews = 35;

        $post = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $firstCountOfViews,
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            DateTimeImmutable::createFromInterface($this->faker->dateTime()),
            countOfViews: $secondCountOfViews,
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'countOfViews' => Range::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'countOfViews' => [
                    'max' => 40,
                ],
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(2, $result);
    }
}
