<?php

declare(strict_types=1);

namespace IQDEV\Tests\Packages\DoctrineHttpFilter;

use IQDEV\Packages\DoctrineHttpFilter\Exception\FilterParameterValueIsNullException;
use IQDEV\Packages\DoctrineHttpFilter\Filter\Like;
use IQDEV\Packages\DoctrineHttpFilter\HttpFilter;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Entity\Post;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Repository\PostRepository;
use Symfony\Component\HttpFoundation\Request;

class FilterByLikeTest extends TestCase
{
    public function testSuccessFilterLikeWithOneResult(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $title = 'Видеопредставление';
        $subtitle = 'Видео';

        $post = new Post(
            $title,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'title' => Like::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => $subtitle,
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(1, $result);
        $this->assertStringContainsString($subtitle, current($result)->title);
    }

    public function testSuccessFilterLikeWithSeveralResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $titleFirst = 'Видеопредставление';
        $titleSecond = 'Видеомагнитофон';
        $subtitle = 'Видео';

        $post = new Post(
            $titleFirst,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $post2 = new Post(
            $titleSecond,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'title' => Like::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => $subtitle,
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(2, $result);
    }

    public function testSuccessFilterLikeWithoutResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $title = 'Видеопредставление';
        $subtitle = 'видео';

        $post = new Post(
            $title,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $this->em->persist($post);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'title' => Like::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => $subtitle,
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertEmpty($result);
    }

    public function testFilterLikeWithoutParameterValue(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $this->expectException(FilterParameterValueIsNullException::class);

        $postRepository->createQueryByFilter([
            'title' => Like::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [],
        ]))
            ->getQuery()
            ->getResult();
    }

    public function testFilterDateWithDifferentKey(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $this->expectException(FilterParameterValueIsNullException::class);

        $postRepository->createQueryByFilter([
            'title' => Like::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'notTitle' => 'Not title value',
            ],
        ]))
            ->getQuery()
            ->getResult();
    }
}
