<?php

declare(strict_types=1);

namespace IQDEV\Tests\Packages\DoctrineHttpFilter;

use IQDEV\Packages\DoctrineHttpFilter\Exception\FilterParameterValueIsNullException;
use IQDEV\Packages\DoctrineHttpFilter\Filter\In;
use IQDEV\Packages\DoctrineHttpFilter\HttpFilter;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Entity\Post;
use IQDEV\Tests\Packages\DoctrineHttpFilter\Repository\PostRepository;
use Symfony\Component\HttpFoundation\Request;

class FilterByInTest extends TestCase
{
    public function testSuccessFilterInWithOneResult(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $title = 'Спорт';
        $inValues = ['Спорт', 'Наука', 'Кулинария'];

        $post = new Post(
            $title,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $post2 = new Post(
            $this->faker->name(),
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'title' => In::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => $inValues,
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(1, $result);
        $this->assertStringContainsString($inValues[0], current($result)->title);
    }

    public function testSuccessFilterInWithSeveralResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $titleFirst = 'Спорт';
        $titleSecond = 'Кулинария';
        $inValues = ['Спорт', 'Наука', 'Кулинария'];

        $post = new Post(
            $titleFirst,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $post2 = new Post(
            $titleSecond,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $this->em->persist($post);
        $this->em->persist($post2);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'title' => In::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => $inValues,
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertNotEmpty($result);
        $this->assertCount(2, $result);
    }

    public function testSuccessFilterInWithoutResults(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $title = 'Бизнес';
        $inValues = ['Спорт', 'Наука', 'Кулинария'];

        $post = new Post(
            $title,
            $this->faker->text(),
            $this->faker->boolean(),
            \DateTimeImmutable::createFromInterface($this->faker->dateTime()),
        );

        $this->em->persist($post);
        $this->em->flush();

        $result = $postRepository->createQueryByFilter([
            'title' => In::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => $inValues,
            ],
        ]))
            ->getQuery()
            ->getResult();

        $this->assertEmpty($result);
    }

    public function testFilterInWithoutParameterValue(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $this->expectException(FilterParameterValueIsNullException::class);

        $postRepository->createQueryByFilter([
            'title' => In::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [],
        ]))
            ->getQuery()
            ->getResult();
    }

    public function testFilterInWithNotArrayParameterValue(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $this->expectException(FilterParameterValueIsNullException::class);

        $postRepository->createQueryByFilter([
            'title' => In::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'title' => 'not array',
            ],
        ]))
            ->getQuery()
            ->getResult();
    }

    public function testFilterInWithDifferentKey(): void
    {
        /** @var PostRepository $postRepository */
        $postRepository = $this->em->getRepository(Post::class);

        $this->expectException(FilterParameterValueIsNullException::class);

        $postRepository->createQueryByFilter([
            'title' => In::class,
        ], new Request([
            HttpFilter::REQUEST_FILTER_KEY => [
                'notTitle' => 'Not title value',
            ],
        ]))
            ->getQuery()
            ->getResult();
    }
}
