<?php

namespace App\Entity;

use App\Repository\QuestRepository;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Serializer\Annotation\Groups;

#[ORM\Entity(repositoryClass: QuestRepository::class)]
class Quest
{
    #[ORM\Id]
    #[ORM\GeneratedValue(strategy: 'SEQUENCE')]
    #[ORM\Column]
    private ?int $id = null;

    #[ORM\Column(length: 255)]
    private ?string $name = null;

    #[ORM\Column(type: Types::TEXT)]
    private ?string $short_description = null;

    #[ORM\Column(type: Types::TEXT, nullable: true)]
    private ?string $full_description = null;

    #[ORM\Column(type: Types::DATETIME_MUTABLE)]
    private ?\DateTimeInterface $date = null;

    #[ORM\Column(type: Types::DATETIME_MUTABLE)]
    private ?\DateTimeInterface $final_date = null;

    #[ORM\Column(nullable: true)]
    private ?int $max_appointments = null;

    /**
     * @var Collection<int, QuestImage>
     */
    #[ORM\OneToMany(targetEntity: QuestImage::class, mappedBy: 'quest')]
    private Collection $gallery;

    /**
     * @var Collection<int, Appointment>
     */
    #[ORM\OneToMany(targetEntity: Appointment::class, mappedBy: 'quest')]
    private Collection $appointments;

    /**
     * @var Collection<int, Review>
     */
    #[ORM\OneToMany(targetEntity: Review::class, mappedBy: 'quest')]
    private Collection $reviews;

    /**
     * @var Collection<int, Tag>
     */
    #[ORM\ManyToMany(targetEntity: Tag::class, mappedBy: 'quests')]
    private Collection $tags;

    #[ORM\ManyToOne]
    private ?Theme $theme = null;

    #[ORM\ManyToOne]
    private ?Genre $genre = null;

    public function __construct()
    {
        $this->gallery = new ArrayCollection();
        $this->appointments = new ArrayCollection();
        $this->reviews = new ArrayCollection();
        $this->tags = new ArrayCollection();
    }

    #[Groups(['all', 'card', 'detail', 'profile'])]
    public function getId(): ?int
    {
        return $this->id;
    }

    #[Groups(['all', 'card', 'detail', 'profile'])]
    public function getName(): ?string
    {
        return $this->name;
    }

    public function setName(string $name): static
    {
        $this->name = $name;

        return $this;
    }

    #[Groups(['all', 'card', 'profile'])]
    public function getShortDescription(): ?string
    {
        return $this->short_description;
    }

    public function setShortDescription(string $short_description): static
    {
        $this->short_description = $short_description;

        return $this;
    }

    #[Groups(['all', 'detail'])]
    public function getFullDescription(): ?string
    {
        return $this->full_description;
    }

    public function setFullDescription(?string $full_description): static
    {
        $this->full_description = $full_description;

        return $this;
    }

    #[Groups(['all', 'card', 'profile'])]
    public function getDate(): ?\DateTimeInterface
    {
        return $this->date;
    }

    public function setDate(\DateTimeInterface $date): static
    {
        $this->date = $date;

        return $this;
    }

    #[Groups(['all', 'detail'])]
    public function getFinalDate(): ?\DateTimeInterface
    {
        return $this->final_date;
    }

    public function setFinalDate(\DateTimeInterface $final_date): static
    {
        $this->final_date = $final_date;

        return $this;
    }

    #[Groups(['all', 'detail'])]
    public function getMaxAppointments(): ?int
    {
        return $this->max_appointments;
    }

    public function setMaxAppointments(?int $max_appointments): static
    {
        $this->max_appointments = $max_appointments;

        return $this;
    }

    /**
     * @return Collection<int, QuestImage>
     */
    #[Groups(['all', 'card', 'detail', 'profile'])]
    public function getGallery(): Collection
    {
        return $this->gallery;
    }

    public function addGallery(QuestImage $gallery): static
    {
        if (!$this->gallery->contains($gallery)) {
            $this->gallery->add($gallery);
            $gallery->setQuest($this);
        }

        return $this;
    }

    public function removeGallery(QuestImage $gallery): static
    {
        if ($this->gallery->removeElement($gallery)) {
            // set the owning side to null (unless already changed)
            if ($gallery->getQuest() === $this) {
                $gallery->setQuest(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Appointment>
     */
    #[Groups(['all'])]
    public function getAppointments(): Collection
    {
        return $this->appointments;
    }

    public function addAppointment(Appointment $appointment): static
    {
        if (!$this->appointments->contains($appointment)) {
            $this->appointments->add($appointment);
            $appointment->setQuest($this);
        }

        return $this;
    }

    #[Groups(['all', 'detail'])]
    public function getAppointmentCount(): int
    {
        return count($this->appointments);
    }

    public function removeAppointment(Appointment $appointment): static
    {
        if ($this->appointments->removeElement($appointment)) {
            // set the owning side to null (unless already changed)
            if ($appointment->getQuest() === $this) {
                $appointment->setQuest(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Review>
     */
    #[Groups(['all', 'detail'])]
    public function getReviews(): Collection
    {
        return $this->reviews;
    }

    public function addReview(Review $review): static
    {
        if (!$this->reviews->contains($review)) {
            $this->reviews->add($review);
            $review->setQuest($this);
        }

        return $this;
    }

    public function removeReview(Review $review): static
    {
        if ($this->reviews->removeElement($review)) {
            // set the owning side to null (unless already changed)
            if ($review->getQuest() === $this) {
                $review->setQuest(null);
            }
        }

        return $this;
    }

    #[Groups(['all', 'card', 'detail', 'profile'])]
    public function getRating(): float
    {
        $count = count($this->reviews);
        $rating = 0;
        foreach ($this->reviews as $review) {
            $rating += $review->getRating() ?: 0;
        }

        if ($count > 0) {
            return $rating / $count;
        }

        return 0;
    }

    /**
     * @return Collection<int, Tag>
     */
    #[Groups(['all', 'card', 'detail'])]
    public function getTags(): Collection
    {
        return $this->tags;
    }

    public function addTag(Tag $tag): static
    {
        if (!$this->tags->contains($tag)) {
            $this->tags->add($tag);
            $tag->addQuest($this);
        }

        return $this;
    }

    public function removeTag(Tag $tag): static
    {
        if ($this->tags->removeElement($tag)) {
            $tag->removeQuest($this);
        }

        return $this;
    }

    #[Groups(['all', 'card', 'detail'])]
    public function getTheme(): ?Theme
    {
        return $this->theme;
    }

    public function setTheme(?Theme $theme): static
    {
        $this->theme = $theme;

        return $this;
    }

    #[Groups(['all', 'card', 'detail'])]
    public function getGenre(): ?Genre
    {
        return $this->genre;
    }

    public function setGenre(?Genre $genre): static
    {
        $this->genre = $genre;

        return $this;
    }
}
