<?php

namespace App\Service;

use App\Entity\User;
use App\Redis\Redis;
use Doctrine\Persistence\ManagerRegistry;
use Psr\Cache\InvalidArgumentException;

class RedisToken
{
    private int $ttl;
    private Redis $redis;

    public function __construct(
        int $ttl,
        private ManagerRegistry $doctrine
    )
    {
        $this->ttl = $ttl;
        $this->redis = Redis::getInstance();
    }

    /**
     * Установка времени создания токена
     *
     * @param string $username
     *
     * @return int
     *
     * @throws InvalidArgumentException
     */
    public function set(string $username): int
    {
        $time = time();
        $this->redis->set($this->format($this->getId($username)), $time);
        return $time;
    }

    /**
     * Получение времени
     *
     * @param string $username
     *
     * @return bool
     *
     * @throws InvalidArgumentException
     */
    public function check(string $username): bool
    {
        $time = $this->redis->get($this->format($this->getId($username)));

        if (!$time) {
            $time = $this->set($username);
        }

        $time += $this->ttl;

        $now = time();

        if ($now > $time) {
            $this->delete($username);
            return false;
        }

        $this->set($username);

        return true;
    }

    /**
     * Удаление времени
     *
     * @param string $username
     *
     * @return void
     */
    public function delete(string $username): void
    {
        if ($id = $this->getId($username)) {
            $this->redis->delete($this->format($id));
        }
    }

    /**
     * Очистка прошедшего времени
     *
     * @return void
     *
     * @throws InvalidArgumentException
     */
    public function clearOld(): void
    {
        $now = time();

        $users = $this->doctrine->getManager()->getRepository(User::class)->findAll();
        foreach ($users as $user) {
            $code = $this->format($user->getId());
            if ($this->redis->has($code)) {
                $time = $this->redis->get($code);

                if ($now > $time) {
                    $this->redis->delete($code);
                }
            }
        }
    }

    private function format(int $id): string
    {
        return 'token_ttl_' . $id;
    }

    private function getId(string $username): ?int
    {
        $user = $this->doctrine->getManager()->getRepository(User::class)->findOneBy(['email' => $username]);
        if ($user) {
            return $user->getId();
        }

        return null;
    }
}