<?php

namespace App\Messenger\Handler;

use App\Entity\Quest;
use App\Messenger\Message\QuestMessage;
use App\Messenger\Objects\QuestsStart;
use DateInterval;
use Doctrine\Common\Collections\Criteria;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;
use Symfony\Component\Messenger\MessageBusInterface;

#[AsMessageHandler]
class QuestStartMessageHandler
{
    public function __construct(
        protected MessageBusInterface $bus,
        private ManagerRegistry       $doctrine,
        private string $fromEmail,
    )
    {
    }

    /**
     * Обработка письма из очереди
     *
     * @param QuestsStart $message
     *
     * @return void
     */
    public function __invoke(QuestsStart $message): void
    {
        $quests = $this->doctrine->getRepository(Quest::class)->getByCriteria($this->getCriteria());

        foreach ($quests as $quest) {
            $appointments = $quest->getAppointments();
            $questName = $quest->getName();
            foreach ($appointments as $appointment) {
                $user = $appointment->getRelatedUser();
                if ($user) {
                    $userName = $user->getFullName();
                    $questMessage = new QuestMessage(
                        $this->fromEmail,
                        $user->getEmail(),
                        "Квест \"{$questName}\" стартует уже через 3 дня!",
                        <<<HTML
                        <div>
                            Здравствуйте, {$userName}!
                        </div>
                        <div>
                            Квест "{$questName}" стартует уже через 3 дня!
                        </div>
                        <div>
                            Чтобы не забыть, запланируйте его у себя в календаре!
                        </div>
                        HTML
                    );
                    $this->bus->dispatch($questMessage);
                }
            }
        }
    }

    /**
     * Фильтр квестов
     *
     * @return Criteria
     */
    public function getCriteria(): Criteria
    {
        $criteria = Criteria::create();
        $expr = Criteria::expr();

        if (!$expr) {
            return $criteria;
        }

        $startDate = new \DateTime();
        $startDate->setTime(0,0);
        $startDate->add(new DateInterval('P3D'));
        $endDate = new \DateTime();
        $endDate->setTime(23,59, 59);
        $endDate->add(new DateInterval('P3D'));

        $criteria
            ->andWhere($expr->gte('date', $startDate))
            ->andWhere($expr->lte('date', $endDate));

        return $criteria;
    }
}