<?php

namespace App\Listeners;

use App\Entity\User;
use App\Service\RedisToken;
use App\Service\Response\Classes\TokenResponse;
use JsonException;
use Lexik\Bundle\JWTAuthenticationBundle\Event\AuthenticationFailureEvent;
use Lexik\Bundle\JWTAuthenticationBundle\Event\AuthenticationSuccessEvent;
use Lexik\Bundle\JWTAuthenticationBundle\Event\JWTCreatedEvent;
use Lexik\Bundle\JWTAuthenticationBundle\Event\JWTDecodedEvent;
use Lexik\Bundle\JWTAuthenticationBundle\Event\JWTExpiredEvent;
use Lexik\Bundle\JWTAuthenticationBundle\Event\JWTInvalidEvent;
use Lexik\Bundle\JWTAuthenticationBundle\Event\JWTNotFoundEvent;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\HttpFoundation\Response;

class JwtListener
{
    public function __construct(
        private RedisToken $redisToken,
    )
    {
    }

    /**
     * @param AuthenticationSuccessEvent $event
     *
     * @return void
     *
     * @throws JsonException
     */
    public function onAuthenticationSuccessResponse(AuthenticationSuccessEvent $event): void
    {
        $data = $event->getData();
        $user = $event->getUser();

        if (!$user instanceof User) {
            return;
        }

        $response = new TokenResponse();

        if ($user->isDeleted()) {
            $response->addError('Пользователь удален');
        } else {
            $response->setToken($data['token']);
        }

        $data = json_decode($response->getResponse()->getContent(), true, 512, JSON_THROW_ON_ERROR);
        $event->setData($data);
    }

    /**
     * @param AuthenticationFailureEvent $event
     *
     * @throws JsonException
     */
    public function onAuthenticationFailureResponse(AuthenticationFailureEvent $event): void
    {
        $response = new \App\Service\Response\Classes\Response();
        $response->addError('Неверный email или пароль');
        $response->setStatusCode(Response::HTTP_UNAUTHORIZED);
        $event->setResponse($response->getResponse());
    }

    /**
     * @param JWTInvalidEvent $event
     *
     * @throws JsonException
     */
    public function onJWTInvalid(JWTInvalidEvent $event): void
    {
        $response = new \App\Service\Response\Classes\Response();
        $response->addError('Неверный токен авторизации');
        $response->setStatusCode(Response::HTTP_FORBIDDEN);

        $event->setResponse($response->getResponse());
    }

    /**
     * @param JWTNotFoundEvent $event
     *
     * @throws JsonException
     */
    public function onJWTNotFound(JWTNotFoundEvent $event): void
    {
        $response = new \App\Service\Response\Classes\Response();
        $response->addError('Отсутствует токен');
        $response->setStatusCode(Response::HTTP_UNAUTHORIZED);

        $event->setResponse($response->getResponse());
    }

    /**
     * @param JWTExpiredEvent $event
     *
     * @throws JsonException
     */
    public function onJWTExpired(JWTExpiredEvent $event): void
    {
        $response = new \App\Service\Response\Classes\Response();
        $response->addError('Срок действия вашего токена истек, пожалуйста, обновите его');
        $response->setStatusCode(Response::HTTP_FORBIDDEN);

        $event->setResponse($response->getResponse());

        $this->redisToken->clearOld();
    }

    /**
     * @param JWTCreatedEvent $event
     *
     * @return void
     *
     * @throws InvalidArgumentException
     */
    public function onJWTCreated(JWTCreatedEvent $event): void
    {
        $payload = $event->getData();

        $this->redisToken->set($payload['username']);
    }

    /**
     * @param JWTDecodedEvent $event
     *
     * @return void
     *
     * @throws InvalidArgumentException
     */
    public function onJWTDecoded(JWTDecodedEvent $event): void
    {
        if ($event->isValid()) {
            $payload = $event->getPayload();

            if (!$this->redisToken->check($payload['username'])) {
                $event->markAsInvalid();
            }

            $event->setPayload($payload);
        }
    }
}