<?php

namespace App\Service\Send\Classes;

use App\Entity\User;
use App\Entity\UserCode;
use App\Service\Response\ResponseServiceInterface;
use App\Service\Send\SendService;
use App\Service\Send\SendServiceInterface;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Component\DependencyInjection\Attribute\AsAlias;
use Symfony\Component\Serializer\SerializerInterface;

#[AsAlias]
class CodeSendService implements SendServiceInterface
{
    private ?User $user = null;

    private ?ResponseServiceInterface $response = null;

    public function __construct(
        private ManagerRegistry $doctrine,
        private SendService     $sendService,
        private SerializerInterface      $serializer
    )
    {
    }

    public function setResponse(?ResponseServiceInterface $response): void
    {
        $this->response = $response;
    }

    public function setUser(?User $user): void
    {
        $this->user = $user;
    }

    public function getSubject(): string
    {
        return '';
    }

    public function getBody(): string
    {
        return '<span id="code">{code}</span>';
    }

    public function send(): void
    {
        if ($this->user === null) {
            $this->response->addError('Письмо не отправлено, пользователь не получен');
            return;
        }
        $serializedUser = $this->serializer->serialize($this->user, 'json', ['groups' => ['profile']]);
        $values = json_decode($serializedUser, true, 512, JSON_THROW_ON_ERROR) ?: [];

        $codeObj = $this->user->getRegisterCode();
        $code = null;
        $time = null;
        if ($codeObj === null) {
            $codeObj = new UserCode();
            $codeObj->setRelatedUser($this->user);
        }

        try {
            $om = $this->doctrine->getManager();
            $om->persist($codeObj);
            $om->flush();
            $code = $codeObj->getCode();
            $date = $codeObj->getDate();
            $time = $date?->diff(new \DateTime());
        } catch (\Exception $exception) {
            $this->response->addError('Ошибка генерации кода');
        }

        if ($code) {
            $values['code'] = $code;
            $timeStr = 'нет';
            if ($time) {
                $timeStr = $time->format('%H:%I:%S');
            }
            $values['time'] = $timeStr;
            $this->sendService->setTo($this->user->getEmail());
            $this->sendService->setSubject($this->formatSubject($values));
            $this->sendService->setBody($this->formatBody($values));
            $this->sendService->send();
            $this->response->addMessage('Письмо с кодом отправлено');
        } else {
            $this->response->addError('Ошибка генерации кода');
        }
    }

    /**
     * Подстановка значений в письмо
     *
     * @param array $values
     *
     * @return string
     */
    private function formatBody(array $values): string
    {
        $body = $this->getBody();

        return self::textReplace($body, $values);
    }

    /**
     * Подстановка значений в тему письма
     *
     * @param array $values
     *
     * @return string
     */
    private function formatSubject(array $values): string
    {
        $subject = $this->getSubject();

        return self::textReplace($subject, $values);
    }

    /**
     * Замена переменных текста
     *
     * @param string $text
     * @param array $values
     *
     * @return string
     */
    private static function textReplace(string $text, array $values): string
    {
        foreach ($values as $name => $value) {
            if (is_string($value)) {
                $text = str_replace('{' . $name . '}', $value, $text);
            }
        }

        $match = [];
        preg_match('/{\w+}/', $text, $match);
        foreach ($match as $value) {
            $text = str_replace($value, '', $text);
        }

        return $text;
    }
}