<?php

namespace App\Service\Action\Classes;

use App\Entity\UserHistory;
use App\Entity\UserImage;
use App\Service\Action\UserBaseActionService;
use App\Service\Dto\Classes\ImageDto;
use App\Service\Dto\DtoServiceInterface;
use Symfony\Component\DependencyInjection\Attribute\AsAlias;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\File\Exception\FileException;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\Mime\MimeTypes;
use Symfony\Component\String\Slugger\SluggerInterface;
use Symfony\Contracts\Service\Attribute\Required;

#[AsAlias(id: 'action.profile.image.save', public: true)]
class SaveImage extends UserBaseActionService
{
    // Доступные расширения файлов
    public const IMAGE_EXTENSIONS = [
        'jpg',
        'png'
    ];

    public function __construct(
        private string                   $targetDirectory,
        private SluggerInterface         $slugger
    )
    {
    }

    #[Required] public function initDto(
        #[Autowire(service: 'dto.image')]
        DtoServiceInterface $dtoService
    ): void
    {
        parent::initDto($dtoService);
    }


    public function runAction(): void
    {
        $file = $this->saveFile();
        if ($file) {
            $em = $this->doctrine->getManager();

            $oldImage = $this->user->getImage();
            if ($oldImage) {
                $oldImage->setName($file->getName());
                $oldImage->setType($file->getType());
                $oldImage->setPath($file->getPath());
                $file = $oldImage;
            } else {
                $newUserHistory = new UserHistory();
                $newUserHistory->setType(UserHistory::TYPE_CREATE);
                $newUserHistory->setField('image');
                $newUserHistory->setValue($file->getName());
                $this->user->addUserHistory($newUserHistory);
                $this->user->setImage($file);
            }
            try {
                $em->persist($file);
                $em->flush();
                $this->responseService->addMessage('Изображение сохранено');
            } catch (\Exception $exception) {
                $this->responseService->addError('Ошибка сохранения файла пользователя');
            }
        } else {
            $this->responseService->addError('Ошибка сохранения файла');
        }
    }

    public function saveFile(): ?UserImage
    {
        /** @var ImageDto $dto */
        $dto = $this->getDto();

        $matches = [];
        if (!preg_match('/^data:([a-z0-9][a-z0-9\!\#\$\&\-\^\_\+\.]{0,126}\/[a-z0-9][a-z0-9\!\#\$\&\-\^\_\+\.]{0,126}(;[a-z0-9\-]+\=[a-z0-9\-]+)?)?(;base64)?,([a-z0-9\!\$\&\\\'\,\(\)\*\+\,\;\=\-\.\_\~\:\@\/\?\%\s]*\s*$)/i', $dto->data ?: '', $matches)) {
            return null;
        }
        $extension = $matches[1];
        $content = $matches[4];
        if ($extension && $content) {
            $mimeTypes = new MimeTypes();
            $types = $mimeTypes->getExtensions($extension);
            if (empty($types)) {
                $this->responseService->addError('Неизвестное расширения файла');
                return null;
            }
            if (empty(array_intersect($types, self::IMAGE_EXTENSIONS))) {
                $this->responseService->addError('Файл расширения "'. reset($types) .'" недоступен для загрузки. Доступные расширения: ' . implode(', ', self::IMAGE_EXTENSIONS) . '.');
                return null;
            }

            $filename = pathinfo($dto->name, PATHINFO_FILENAME);

            $filename = $filename . '.' . reset($types);

            $decoded = base64_decode($content);
            if (!$decoded) {
                $this->responseService->addError('Ошибка декодирования файла');
                return null;
            }

            $tmpPath = sys_get_temp_dir() . '/file_upload' . uniqid();

            if (file_put_contents($tmpPath, base64_decode($content))) {
                $uploadFile = new UploadedFile($tmpPath, $filename, $extension, null, true);

                $originalFilename = pathinfo($uploadFile->getClientOriginalName(), PATHINFO_FILENAME);
                $safeFilename = $this->slugger->slug($originalFilename);
                $fileName = $safeFilename . '-' . uniqid() . '.' . $uploadFile->guessExtension();
                $filedir = $this->targetDirectory;
                try {
                    $file = $uploadFile->move($filedir, $fileName);
                    $dmFile = new UserImage();
                    $dmFile->setName($uploadFile->getClientOriginalName());
                    $dmFile->setPath($file->getRealPath());
                    $dmFile->setType($extension);
                    return $dmFile;
                } catch (FileException $e) {
                    return null;
                }
            }
        }

        return null;
    }

    public function needDto(): bool
    {
        return true;
    }

    public function checkDelete(): bool
    {
        return true;
    }

    public function checkConfirm(): bool
    {
        return false;
    }
}