<?php

namespace App\Listeners;

use App\Entity\User;
use App\Entity\UserHistory;
use Doctrine\Bundle\DoctrineBundle\Attribute\AsEntityListener;
use Doctrine\ORM\Event\PreFlushEventArgs;
use Doctrine\ORM\Event\PreUpdateEventArgs;
use Doctrine\ORM\Events;
use Doctrine\ORM\UnitOfWork;
use Doctrine\Persistence\ObjectManager;
use ReflectionClass;

#[AsEntityListener(event: Events::preFlush, method: 'prePersist', entity: User::class)]
#[AsEntityListener(event: Events::preUpdate, method: 'preUpdate', entity: User::class)]
#[AsEntityListener(event: Events::preFlush, method: 'prePersistHistory', entity: UserHistory::class)]
#[AsEntityListener(event: Events::preUpdate, method: 'preUpdateHistory', entity: UserHistory::class)]
class UserListener
{
    public const HISTORY_FIELDS = [
        'email' => 'Email',
        'name' => 'Имя',
        'surname' => 'Фамилия',
        'patronymic' => 'Отчество',
        'phone_number' => 'Номер телефона'
    ];

    public function prePersist(User $user, PreFlushEventArgs $args): void
    {
        $this->checkEmail($user, $args->getObjectManager());
        $this->saveHistory($user, $args->getObjectManager());
    }

    public function preUpdate(User $user, PreUpdateEventArgs $args): void
    {
        $this->checkEmail($user, $args->getObjectManager());
        $this->saveHistory($user, $args->getObjectManager());
    }

    public function saveHistory(User $user, ObjectManager $om): void
    {
        /** @var UnitOfWork $uow */
        $uow = $om->getUnitOfWork();
        $originalValues = $uow->getOriginalEntityData($user);
        $userExists = null;
        if ($originalValues) {
            $userExists = User::createByArray($originalValues);
        }
        $checkUser = $user->newCopy();

        if ($image = $user->getImage()) {
            $imageOriginalValues = $uow->getOriginalEntityData($user->getImage());
            if ($imageOriginalValues) {
                if ($image->getPath() !== $imageOriginalValues['path']) {
                    $newUserHistory = new UserHistory();
                    $newUserHistory->setType(UserHistory::TYPE_UPDATE);
                    $newUserHistory->setField('image');
                    $newUserHistory->setValue($image->getName());
                    $newUserHistory->setOldValue($imageOriginalValues['name']);
                    $user->addUserHistory($newUserHistory);
                }
            }
        }

        if ($user->getId()) {
            if ($userExists) {
                $reflectionClass = new ReflectionClass($user);
                foreach ($reflectionClass->getProperties() as $property) {
                    $name = $property->getName();
                    if (!isset(self::HISTORY_FIELDS[$name])) {
                        continue;
                    }
                    $value = $property->getValue($checkUser);
                    $oldValue = $property->getValue($userExists);
                    if ($value !== $oldValue) {
                        if (empty($value) && empty($oldValue)) {
                            continue;
                        }

                        $type = UserHistory::TYPE_UPDATE;
                        if (empty($value)) {
                            $type = UserHistory::TYPE_DELETE;
                        } elseif (empty($oldValue)) {
                            $type = UserHistory::TYPE_CREATE;
                        }
                        $newUserHistory = new UserHistory();
                        $newUserHistory->setType($type);
                        $newUserHistory->setField($name);
                        $newUserHistory->setValue($value);
                        $newUserHistory->setOldValue($oldValue);
                        $user->addUserHistory($newUserHistory);
                    }
                }
                if ($userExists->isDeleted() !== $user->isDeleted()) {
                    $newUserHistory = new UserHistory();
                    $newUserHistory->setType($user->isDeleted() ? UserHistory::TYPE_DELETE : UserHistory::TYPE_RECOVERY);
                    $newUserHistory->setField('user');
                    $user->addUserHistory($newUserHistory);
                }
                if ($userExists->isConfirm() !== $user->isConfirm()) {
                    $newUserHistory = new UserHistory();
                    $newUserHistory->setType($user->isConfirm() ? UserHistory::TYPE_CREATE : UserHistory::TYPE_DELETE);
                    if ($user->isConfirm()) {
                        $newUserHistory->setValue($user->getEmail());
                    }
                    $newUserHistory->setField('confirm');
                    $user->addUserHistory($newUserHistory);
                }
                if ($userExists->getPassword() !== $user->getPassword()) {
                    $newUserHistory = new UserHistory();
                    $newUserHistory->setType(UserHistory::TYPE_UPDATE);
                    $newUserHistory->setField('password');
                    $user->addUserHistory($newUserHistory);
                }
            }
        } else {
            $newUserHistory = new UserHistory();
            $newUserHistory->setType(UserHistory::TYPE_CREATE);
            $newUserHistory->setField('user');
            $user->addUserHistory($newUserHistory);
        }
    }

    public function checkEmail(User $user, ObjectManager $om): void
    {
        if ($user->getId()) {
            /** @var UnitOfWork $uow */
            $uow = $om->getUnitOfWork();
            $originalValues = $uow->getOriginalEntityData($user);
            $userExists = null;
            if ($originalValues) {
                $userExists = User::createByArray($originalValues);
            }
            if ($userExists) {
                if ($userExists->getEmail() !== $user->getEmail()) {
                    $user->setConfirm($user->getLastConfirmEmail() === $user->getEmail());
                }
            }
        } else {
            $user->setConfirm(false);
        }
    }

    public function prePersistHistory(UserHistory $userHistory, PreFlushEventArgs $args): void
    {
        $this->setHistoryDate($userHistory);
    }

    public function preUpdateHistory(UserHistory $userHistory, PreUpdateEventArgs $args): void
    {
        $this->setHistoryDate($userHistory);
    }

    public function setHistoryDate(UserHistory $userHistory): void
    {
        if (!$userHistory->getDate()) {
            $userHistory->setDate(new \DateTime());
        }
    }
}