<?php

namespace App\Entity;

use App\Repository\UserRepository;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use ReflectionClass;
use Symfony\Component\PropertyInfo\Extractor\ReflectionExtractor;
use Symfony\Component\Security\Core\User\PasswordAuthenticatedUserInterface;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Serializer\Annotation\Ignore;
use Symfony\Component\Serializer\Encoder\JsonEncoder;
use Symfony\Component\Serializer\Mapping\Factory\ClassMetadataFactory;
use Symfony\Component\Serializer\Mapping\Loader\AttributeLoader;
use Symfony\Component\Serializer\NameConverter\CamelCaseToSnakeCaseNameConverter;
use Symfony\Component\Serializer\Normalizer\DateTimeNormalizer;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;
use Symfony\Component\Serializer\Serializer;
use Symfony\Component\Validator\Constraints as Assert;

#[ORM\Entity(repositoryClass: UserRepository::class)]
#[ORM\Table(name: '`user`')]
#[ORM\UniqueConstraint(name: 'UNIQ_IDENTIFIER_EMAIL', fields: ['email'])]
class User implements UserInterface, PasswordAuthenticatedUserInterface
{
    #[ORM\Id]
    #[ORM\GeneratedValue(strategy: 'SEQUENCE')]
    #[ORM\Column]
    private ?int $id = null;

    #[ORM\Column(length: 180)]
    private ?string $email = null;

    /**
     * @var list<string> The user roles
     */
    #[ORM\Column]
    private array $roles = [];

    /**
     * @var string The hashed password
     */
    #[ORM\Column]
    private ?string $password = null;

    #[ORM\Column(length: 255)]
    private ?string $name = null;

    #[ORM\Column(length: 255)]
    private ?string $surname = null;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $patronymic = null;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $phone_number = null;

    #[ORM\OneToOne(mappedBy: 'related_user', cascade: ['persist', 'remove'])]
    private ?UserImage $image = null;

    #[ORM\Column]
    private ?bool $confirm = null;

    #[ORM\Column]
    private ?bool $deleted = null;

    #[ORM\OneToOne(mappedBy: 'related_user', cascade: ['persist', 'remove'])]
    private ?UserCode $register_code = null;

    /**
     * @var Collection<int, UserHistory>
     */
    #[ORM\OneToMany(targetEntity: UserHistory::class, mappedBy: 'related_user', cascade: ['persist', 'remove'],fetch: 'EAGER')]
    private Collection $userHistories;

    /**
     * @var Collection<int, Appointment>
     */
    #[ORM\OneToMany(targetEntity: Appointment::class, mappedBy: 'related_user')]
    private Collection $appointments;

    /**
     * @var Collection<int, Review>
     */
    #[ORM\OneToMany(targetEntity: Review::class, mappedBy: 'related_user')]
    private Collection $reviews;

    /**
     * @var Collection<int, Like>
     */
    #[ORM\OneToMany(targetEntity: Like::class, mappedBy: 'related_user')]
    private Collection $likes;

    /**
     * @var Collection<int, Favorite>
     */
    #[ORM\OneToMany(targetEntity: Favorite::class, mappedBy: 'related_user')]
    private Collection $favorites;

    public function __construct()
    {
        $this->userHistories = new ArrayCollection();
        $this->appointments = new ArrayCollection();
        $this->reviews = new ArrayCollection();
        $this->likes = new ArrayCollection();
        $this->favorites = new ArrayCollection();
    }

    #[Groups(['all'])]
    public function getId(): ?int
    {
        return $this->id;
    }

    #[Groups(['all', 'profile', 'edit', 'card', 'detail', 'listen'])]
    public function getEmail(): ?string
    {
        return $this->email;
    }

    public function setEmail(string $email): static
    {
        $this->email = $email;

        return $this;
    }

    /**
     * A visual identifier that represents this user.
     *
     * @see UserInterface
     */
    #[Ignore]
    public function getUserIdentifier(): string
    {
        return (string) $this->email;
    }

    /**
     * @see UserInterface
     *
     * @return list<string>
     */
    #[Groups(['all'])]
    public function getRoles(): array
    {
        $roles = $this->roles;
        // guarantee every user at least has ROLE_USER
        if ($this->isDeleted()) {
            $roles[] = 'ROLE_DELETED';
        } else {
            $roles[] = 'ROLE_USER';
        }

        return array_unique($roles);
    }

    /**
     * @param list<string> $roles
     */
    public function setRoles(array $roles): static
    {
        $this->roles = $roles;

        return $this;
    }

    /**
     * @see PasswordAuthenticatedUserInterface
     */
    #[Groups(['all', 'listen'])]
    public function getPassword(): string
    {
        return $this->password;
    }

    public function setPassword(string $password): static
    {
        $this->password = $password;

        return $this;
    }

    /**
     * @see UserInterface
     */
    public function eraseCredentials(): void
    {
        // If you store any temporary, sensitive data on the user, clear it here
        // $this->plainPassword = null;
    }

    #[Groups(['all', 'profile', 'edit', 'listen'])]
    public function getName(): ?string
    {
        return $this->name;
    }

    public function setName(string $name): static
    {
        $this->name = $name;

        return $this;
    }

    #[Groups(['all', 'profile', 'edit', 'listen'])]
    public function getSurname(): ?string
    {
        return $this->surname;
    }

    public function setSurname(string $surname): static
    {
        $this->surname = $surname;

        return $this;
    }

    #[Groups(['all', 'profile', 'edit', 'listen'])]
    public function getPatronymic(): ?string
    {
        return $this->patronymic;
    }

    public function setPatronymic(?string $patronymic): static
    {
        $this->patronymic = $patronymic;

        return $this;
    }

    #[Groups(['all', 'profile', 'edit', 'listen'])]
    public function getPhoneNumber(): ?string
    {
        return $this->phone_number;
    }

    public function setPhoneNumber(?string $phone_number): static
    {
        $this->phone_number = $phone_number;

        return $this;
    }

    #[Groups(['all', 'profile', 'edit', 'listen'])]
    public function getImage(): ?UserImage
    {
        return $this->image;
    }

    public function setImage(?UserImage $image): static
    {
        // unset the owning side of the relation if necessary
        if ($image === null && $this->image !== null) {
            $this->image->setRelatedUser(null);
        }

        // set the owning side of the relation if necessary
        if ($image !== null && $image->getRelatedUser() !== $this) {
            $image->setRelatedUser($this);
        }

        $this->image = $image;

        return $this;
    }

    #[Groups(['all', 'profile', 'edit', 'listen'])]
    public function isConfirm(): ?bool
    {
        return $this->confirm;
    }

    public function setConfirm(bool $confirm): static
    {
        $this->confirm = $confirm;

        return $this;
    }

    #[Groups(['all', 'profile', 'edit', 'listen'])]
    public function isDeleted(): ?bool
    {
        return $this->deleted;
    }

    public function setDeleted(bool $deleted): static
    {
        $this->deleted = $deleted;

        return $this;
    }

    #[Groups(['card', 'detail'])]
    public function getFullName(): string
    {
        return $this->getSurname() . ' ' . $this->getName() . ' ' . $this->getPatronymic() ?: '';
    }

    public function getRegisterCode(): ?UserCode
    {
        return $this->register_code;
    }

    public function setRegisterCode(?UserCode $register_code): static
    {
        // unset the owning side of the relation if necessary
        if ($register_code === null && $this->register_code !== null) {
            $this->register_code->setRelatedUser(null);
        }

        // set the owning side of the relation if necessary
        if ($register_code !== null && $register_code->getRelatedUser() !== $this) {
            $register_code->setRelatedUser($this);
        }

        $this->register_code = $register_code;

        return $this;
    }

    /**
     * @return Collection<int, UserHistory>
     */
    #[Groups(['all', 'profile'])]
    public function getUserHistories(): Collection
    {
        return $this->userHistories;
    }

    public function addUserHistory(UserHistory $userHistory): static
    {
        if (!$this->userHistories->contains($userHistory)) {
            $this->userHistories->add($userHistory);
            $userHistory->setRelatedUser($this);
        }

        return $this;
    }

    public function removeUserHistory(UserHistory $userHistory): static
    {
        if ($this->userHistories->removeElement($userHistory)) {
            // set the owning side to null (unless already changed)
            if ($userHistory->getRelatedUser() === $this) {
                $userHistory->setRelatedUser(null);
            }
        }

        return $this;
    }

    #[Groups(['all', 'profile'])]
    public function getLastConfirmEmail(): ?string
    {
        $lastDate = null;
        $lastEmail = null;
        foreach ($this->getUserHistories() as $userHistory) {
            if ($userHistory->getField() === 'confirm' && $userHistory->getDate() && $userHistory->getType() === UserHistory::TYPE_CREATE) {
                if ($lastDate === null || $lastDate->getTimestamp() < $userHistory->getDate()->getTimestamp()) {
                    $lastDate = $userHistory->getDate();
                    $lastEmail = $userHistory->getValue();
                }
            }
        }

        return $lastEmail;
    }

    /**
     * Создание пользователя по массиву
     *
     * @param array $data
     *
     * @return self
     */
    #[Ignore()]
    public static function createByArray(array $data, array $groups = ['listen']): ?self
    {
        try {
            $normalizer = new ObjectNormalizer(
                new ClassMetadataFactory(new AttributeLoader()),
                new CamelCaseToSnakeCaseNameConverter(),
                null,
                new ReflectionExtractor()
            );
            $serializer = new Serializer(
                [new DateTimeNormalizer(), $normalizer],
                [new JsonEncoder()]
            );
            return $serializer->deserialize(
                json_encode($data, JSON_THROW_ON_ERROR),
                __CLASS__,
                'json',
                [
                    'groups' => $groups,
                ]
            );
        } catch (\Exception $exception) {
            return null;
        }
    }

    #[Ignore()]
    public function newCopy(array $groups = ['listen']): ?self
    {
        $normalizer = new ObjectNormalizer(
            new ClassMetadataFactory(new AttributeLoader()),
            new CamelCaseToSnakeCaseNameConverter(),
            null,
            new ReflectionExtractor()
        );
        $serializer = new Serializer([new DateTimeNormalizer(), $normalizer], [new JsonEncoder()]);
        $data = $serializer->serialize($this, 'json', ['groups' => $groups]);
        $array = json_decode($data, true, 512, JSON_THROW_ON_ERROR);
        return self::createByArray($array, $groups);
    }

    /**
     * @return Collection<int, Appointment>
     */
    #[Groups(['all', 'profile'])]
    public function getAppointments(): Collection
    {
        return $this->appointments;
    }

    public function addAppointment(Appointment $appointment): static
    {
        if (!$this->appointments->contains($appointment)) {
            $this->appointments->add($appointment);
            $appointment->setRelatedUser($this);
        }

        return $this;
    }

    public function removeAppointment(Appointment $appointment): static
    {
        if ($this->appointments->removeElement($appointment)) {
            // set the owning side to null (unless already changed)
            if ($appointment->getRelatedUser() === $this) {
                $appointment->setRelatedUser(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Review>
     */
    #[Groups(['all', 'reviews'])]
    public function getReviews(): Collection
    {
        return $this->reviews;
    }

    public function addReview(Review $review): static
    {
        if (!$this->reviews->contains($review)) {
            $this->reviews->add($review);
            $review->setRelatedUser($this);
        }

        return $this;
    }

    public function removeReview(Review $review): static
    {
        if ($this->reviews->removeElement($review)) {
            // set the owning side to null (unless already changed)
            if ($review->getRelatedUser() === $this) {
                $review->setRelatedUser(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Like>
     */
    public function getLikes(): Collection
    {
        return $this->likes;
    }

    public function addLike(Like $like): static
    {
        if (!$this->likes->contains($like)) {
            $this->likes->add($like);
            $like->setRelatedUser($this);
        }

        return $this;
    }

    public function removeLike(Like $like): static
    {
        if ($this->likes->removeElement($like)) {
            // set the owning side to null (unless already changed)
            if ($like->getRelatedUser() === $this) {
                $like->setRelatedUser(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Favorite>
     */
    #[Groups(['all', 'favorites'])]
    public function getFavorites(): Collection
    {
        return $this->favorites;
    }

    public function addFavorite(Favorite $favorite): static
    {
        if (!$this->favorites->contains($favorite)) {
            $this->favorites->add($favorite);
            $favorite->setRelatedUser($this);
        }

        return $this;
    }

    public function removeFavorite(Favorite $favorite): static
    {
        if ($this->favorites->removeElement($favorite)) {
            // set the owning side to null (unless already changed)
            if ($favorite->getRelatedUser() === $this) {
                $favorite->setRelatedUser(null);
            }
        }

        return $this;
    }
}
