<?php

namespace App\Messenger\Handler;

use App\Entity\Quest;
use App\Messenger\Message\QuestMessage;
use App\Messenger\Objects\QuestsEnd;
use DateInterval;
use Doctrine\Common\Collections\Criteria;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;
use Symfony\Component\Messenger\MessageBusInterface;

#[AsMessageHandler]
class QuestEndMessageHandler
{
    public function __construct(
        protected MessageBusInterface $bus,
        private ManagerRegistry       $doctrine,
        private string $fromEmail,
    )
    {
    }

    /**
     * Обработка письма из очереди
     *
     * @param QuestsEnd $message
     *
     * @return void
     */
    public function __invoke(QuestsEnd $message): void
    {
        $quests = $this->doctrine->getRepository(Quest::class)->getByCriteria($this->getCriteria());

        foreach ($quests as $quest) {
            $appointments = $quest->getAppointments();
            $questName = $quest->getName();
            foreach ($appointments as $appointment) {
                $user = $appointment->getRelatedUser();
                if ($user) {
                    $userName = $user->getFullName();
                    $questMessage = new QuestMessage(
                        $this->fromEmail,
                        $user->getEmail(),
                        "Квест \"{$questName}\" прошёл!",
                        <<<HTML
                        <div>
                            Здравствуйте, {$userName}!
                        </div>
                        <div>
                            Квест {$questName} уже прошёл, какие у Вас остались впечатления?
                        </div>
                        <div>
                            Оставьте отзыв на сайте
                        </div>
                        HTML
                    );
                    $this->bus->dispatch($questMessage);
                }
            }
        }
    }

    /**
     * Фильтр квестов
     *
     * @return Criteria
     */
    public function getCriteria(): Criteria
    {
        $criteria = Criteria::create();
        $expr = Criteria::expr();

        if (!$expr) {
            return $criteria;
        }

        $startDate = new \DateTime();
        $startDate->setTime(0,0);
        $startDate->sub(new DateInterval('P1D'));
        $endDate = new \DateTime();
        $endDate->setTime(23,59, 59);
        $endDate->sub(new DateInterval('P1D'));

        $criteria
            ->andWhere($expr->gte('date', $startDate))
            ->andWhere($expr->lte('date', $endDate));

        return $criteria;
    }
}