<?php

namespace App\Entity;

use App\Repository\UserRepository;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use ReflectionClass;
use Symfony\Component\PropertyInfo\Extractor\ReflectionExtractor;
use Symfony\Component\Security\Core\User\PasswordAuthenticatedUserInterface;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Serializer\Annotation\Ignore;
use Symfony\Component\Serializer\Encoder\JsonEncoder;
use Symfony\Component\Serializer\Mapping\Factory\ClassMetadataFactory;
use Symfony\Component\Serializer\Mapping\Loader\AttributeLoader;
use Symfony\Component\Serializer\NameConverter\CamelCaseToSnakeCaseNameConverter;
use Symfony\Component\Serializer\Normalizer\DateTimeNormalizer;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;
use Symfony\Component\Serializer\Serializer;
use Symfony\Component\Validator\Constraints as Assert;

#[ORM\Entity(repositoryClass: UserRepository::class)]
#[ORM\Table(name: '`user`')]
#[ORM\UniqueConstraint(name: 'UNIQ_IDENTIFIER_EMAIL', fields: ['email'])]
class User implements UserInterface, PasswordAuthenticatedUserInterface
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private ?int $id = null;

    #[ORM\Column(length: 180)]
    private ?string $email = null;

    /**
     * @var list<string> The user roles
     */
    #[ORM\Column]
    private array $roles = [];

    /**
     * @var string The hashed password
     */
    #[ORM\Column]
    private ?string $password = null;

    #[ORM\Column(length: 255)]
    private ?string $name = null;

    #[ORM\Column(length: 255)]
    private ?string $surname = null;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $patronymic = null;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $phone_number = null;

    #[ORM\OneToOne(mappedBy: 'related_user', cascade: ['persist', 'remove'])]
    private ?UserImage $image = null;

    #[ORM\Column]
    private ?bool $confirm = null;

    #[ORM\Column]
    private ?bool $deleted = null;

    #[ORM\OneToOne(mappedBy: 'related_user', cascade: ['persist', 'remove'])]
    private ?UserCode $register_code = null;

    /**
     * @var Collection<int, UserHistory>
     */
    #[ORM\OneToMany(targetEntity: UserHistory::class, mappedBy: 'related_user', cascade: ['persist', 'remove'],fetch: 'EAGER')]
    private Collection $userHistories;

    public function __construct()
    {
        $this->userHistories = new ArrayCollection();
    }

    #[Groups(['all'])]
    public function getId(): ?int
    {
        return $this->id;
    }

    #[Groups(['all', 'profile', 'edit', 'card', 'detail'])]
    public function getEmail(): ?string
    {
        return $this->email;
    }

    public function setEmail(string $email): static
    {
        $this->email = $email;

        return $this;
    }

    /**
     * A visual identifier that represents this user.
     *
     * @see UserInterface
     */
    #[Ignore]
    public function getUserIdentifier(): string
    {
        return (string) $this->email;
    }

    /**
     * @see UserInterface
     *
     * @return list<string>
     */
    #[Groups(['all'])]
    public function getRoles(): array
    {
        $roles = $this->roles;
        // guarantee every user at least has ROLE_USER
        $roles[] = 'ROLE_USER';
        if ($this->isDeleted()) {
            $roles[] = 'ROLE_DELETED';
        } else if ($this->isConfirm()) {
            $roles[] = 'ROLE_CONFIRMED';
        } else {
            $roles[] = 'ROLE_NOT_CONFIRMED';
        }

        return array_unique($roles);
    }

    /**
     * @param list<string> $roles
     */
    public function setRoles(array $roles): static
    {
        $this->roles = $roles;

        return $this;
    }

    /**
     * @see PasswordAuthenticatedUserInterface
     */
    #[Groups(['all'])]
    public function getPassword(): string
    {
        return $this->password;
    }

    public function setPassword(string $password): static
    {
        $this->password = $password;

        return $this;
    }

    /**
     * @see UserInterface
     */
    public function eraseCredentials(): void
    {
        // If you store any temporary, sensitive data on the user, clear it here
        // $this->plainPassword = null;
    }

    #[Groups(['all', 'profile', 'edit'])]
    public function getName(): ?string
    {
        return $this->name;
    }

    public function setName(string $name): static
    {
        $this->name = $name;

        return $this;
    }

    #[Groups(['all', 'profile', 'edit'])]
    public function getSurname(): ?string
    {
        return $this->surname;
    }

    public function setSurname(string $surname): static
    {
        $this->surname = $surname;

        return $this;
    }

    #[Groups(['all', 'profile', 'edit'])]
    public function getPatronymic(): ?string
    {
        return $this->patronymic;
    }

    public function setPatronymic(?string $patronymic): static
    {
        $this->patronymic = $patronymic;

        return $this;
    }

    #[Groups(['all', 'profile', 'edit'])]
    public function getPhoneNumber(): ?string
    {
        return $this->phone_number;
    }

    public function setPhoneNumber(?string $phone_number): static
    {
        $this->phone_number = $phone_number;

        return $this;
    }

    #[Groups(['all', 'profile', 'edit'])]
    public function getImage(): ?UserImage
    {
        return $this->image;
    }

    public function setImage(?UserImage $image): static
    {
        // unset the owning side of the relation if necessary
        if ($image === null && $this->image !== null) {
            $this->image->setRelatedUser(null);
        }

        // set the owning side of the relation if necessary
        if ($image !== null && $image->getRelatedUser() !== $this) {
            $image->setRelatedUser($this);
        }

        $this->image = $image;

        return $this;
    }

    #[Groups(['all', 'profile', 'edit'])]
    public function isConfirm(): ?bool
    {
        return $this->confirm;
    }

    public function setConfirm(bool $confirm): static
    {
        $this->confirm = $confirm;

        return $this;
    }

    #[Groups(['all', 'profile', 'edit'])]
    public function isDeleted(): ?bool
    {
        return $this->deleted;
    }

    public function setDeleted(bool $deleted): static
    {
        $this->deleted = $deleted;

        return $this;
    }

    #[Groups(['card', 'detail'])]
    public function getFullName(): string
    {
        return $this->getSurname() . ' ' . $this->getName() . ' ' . $this->getPatronymic() ?: '';
    }

    public function getRegisterCode(): ?UserCode
    {
        return $this->register_code;
    }

    public function setRegisterCode(?UserCode $register_code): static
    {
        // unset the owning side of the relation if necessary
        if ($register_code === null && $this->register_code !== null) {
            $this->register_code->setRelatedUser(null);
        }

        // set the owning side of the relation if necessary
        if ($register_code !== null && $register_code->getRelatedUser() !== $this) {
            $register_code->setRelatedUser($this);
        }

        $this->register_code = $register_code;

        return $this;
    }

    /**
     * @return Collection<int, UserHistory>
     */
    #[Groups(['all', 'profile'])]
    public function getUserHistories(): Collection
    {
        return $this->userHistories;
    }

    public function addUserHistory(UserHistory $userHistory): static
    {
        if (!$this->userHistories->contains($userHistory)) {
            $this->userHistories->add($userHistory);
            $userHistory->setRelatedUser($this);
        }

        return $this;
    }

    public function removeUserHistory(UserHistory $userHistory): static
    {
        if ($this->userHistories->removeElement($userHistory)) {
            // set the owning side to null (unless already changed)
            if ($userHistory->getRelatedUser() === $this) {
                $userHistory->setRelatedUser(null);
            }
        }

        return $this;
    }

    #[Groups(['all', 'profile'])]
    public function getLastConfirmEmail(): ?string
    {
        $lastDate = null;
        $lastEmail = null;
        foreach ($this->getUserHistories() as $userHistory) {
            if ($userHistory->getField() === 'confirm' && $userHistory->getDate() && $userHistory->getType() === UserHistory::TYPE_CREATE) {
                if ($lastDate === null || $lastDate->getTimestamp() < $userHistory->getDate()->getTimestamp()) {
                    $lastDate = $userHistory->getDate();
                    $lastEmail = $userHistory->getValue();
                }
            }
        }

        return $lastEmail;
    }

    /**
     * Создание пользователя по массиву
     *
     * @param array $data
     *
     * @return self
     */
    #[Ignore()]
    public static function createByArray(array $data, array $groups = ['edit']): ?self
    {
        try {
            $normalizer = new ObjectNormalizer(
                new ClassMetadataFactory(new AttributeLoader()),
                new CamelCaseToSnakeCaseNameConverter(),
                null,
                new ReflectionExtractor()
            );
            $serializer = new Serializer(
                [$normalizer, new DateTimeNormalizer()],
                [new JsonEncoder()]
            );
            return $serializer->deserialize(
                json_encode($data, JSON_THROW_ON_ERROR),
                __CLASS__,
                'json',
                [
                    'groups' => $groups,
                ]
            );
        } catch (\Exception $exception) {
            return null;
        }
    }

    #[Ignore()]
    public function newCopy(array $groups = ['edit']): ?self
    {
        $normalizer = new ObjectNormalizer(
            new ClassMetadataFactory(new AttributeLoader()),
            new CamelCaseToSnakeCaseNameConverter(),
            null,
            new ReflectionExtractor()
        );
        $serializer = new Serializer([$normalizer], [new JsonEncoder()]);
        $data = $serializer->serialize($this, 'json', ['groups' => $groups]);
        $array = json_decode($data, true, 512, JSON_THROW_ON_ERROR);
        return self::createByArray($array, $groups);
    }
}
