<?php

namespace App\Listeners;

use App\Service\Response\Classes\TokenResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Security\Core\User\UserInterface;
use Gesdinet\JWTRefreshTokenBundle\Model\RefreshTokenInterface;
use Gesdinet\JWTRefreshTokenBundle\Model\RefreshTokenManagerInterface;
use Gesdinet\JWTRefreshTokenBundle\Request\Extractor\ExtractorInterface;
use Lexik\Bundle\JWTAuthenticationBundle\Event\AuthenticationSuccessEvent;
use Gesdinet\JWTRefreshTokenBundle\Generator\RefreshTokenGeneratorInterface;
use Gesdinet\JWTRefreshTokenBundle\EventListener\AttachRefreshTokenOnSuccessListener as EventListenerAttachRefreshTokenOnSuccessListener;

class JwtRefreshListener extends EventListenerAttachRefreshTokenOnSuccessListener
{
    public function __construct(
        RefreshTokenManagerInterface   $refreshTokenManager,
                                       $ttl,
        RequestStack                   $requestStack,
                                       $tokenParameterName,
                                       $singleUse,
        RefreshTokenGeneratorInterface $refreshTokenGenerator,
        ExtractorInterface             $extractor,
        array                          $cookieSettings,
        bool                           $returnExpiration,
        string                         $returnExpirationParameterName
    )
    {
        parent::__construct(
            $refreshTokenManager,
            $ttl,
            $requestStack,
            $tokenParameterName,
            $singleUse,
            $refreshTokenGenerator,
            $extractor,
            $cookieSettings,
            $returnExpiration,
            $returnExpirationParameterName
        );
    }

    /**
     * @param AuthenticationSuccessEvent $event
     *
     * @return void
     *
     * @throws \JsonException
     */
    public function attachRefreshToken(AuthenticationSuccessEvent $event): void
    {
        $user = $event->getUser();

        if (!$user instanceof UserInterface) {
            return;
        }

        $response = new TokenResponse();

        $data = $event->getData();
        $response->setToken($data['data']['token']);
        $request = $this->requestStack->getCurrentRequest();

        if (null === $request) {
            return;
        }

        $refreshTokenString = $this->extractor->getRefreshToken($request, $this->tokenParameterName);

        if ($refreshTokenString && true === $this->singleUse) {
            $refreshToken = $this->refreshTokenManager->get($refreshTokenString);
            $refreshTokenString = null;

            if ($refreshToken instanceof RefreshTokenInterface) {
                $this->refreshTokenManager->delete($refreshToken);
            }
        }

        if ($refreshTokenString) {
            $response->setRefreshToken($refreshTokenString);
        } else {
            $refreshToken = $this->refreshTokenGenerator->createForUserWithTtl($user, $this->ttl);

            $this->refreshTokenManager->save($refreshToken);
            $refreshTokenString = $refreshToken->getRefreshToken();
            $response->setRefreshToken($refreshTokenString);
        }

        // Set response data
        $data = json_decode($response->getResponse()->getContent(), true, 512, JSON_THROW_ON_ERROR);
        $event->getResponse()->setContent($response->getResponse()->getContent());
        $event->setData($data);
    }
}