<?php

namespace App\Service\Action\Classes;

use App\Entity\User;
use App\Service\Action\BaseActionService;
use App\Service\Dto\DtoServiceInterface;
use App\Service\Send\SendServiceInterface;
use ReflectionClass;
use Symfony\Component\DependencyInjection\Attribute\AsAlias;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Contracts\Service\Attribute\Required;

#[AsAlias(id: 'action.register', public: true)]
class Register extends BaseActionService
{
    /**
     * @param UserPasswordHasherInterface $passwordHasher
     */
    public function __construct(
        private UserPasswordHasherInterface $passwordHasher,
    )
    {
    }

    #[Required] public function initDto(
        #[Autowire(service: 'dto.register')]
        DtoServiceInterface $dtoService
    ): void
    {
        parent::initDto($dtoService);
    }

    #[Required] public function initSend(
        #[Autowire(service: 'send.code.register')]
        SendServiceInterface $sendService
    ): void
    {
        parent::initSend($sendService);
    }


    /**
     * Регистрация
     *
     * @return void
     */
    public function runAction(): void
    {
        $dto = $this->getDto();
        $user = $this->createUser();
        if ($user !== null && $dto) {
            $userExists = $this->doctrine->getRepository(User::class)
                ->findOneByUniq($user->getEmail(), $user->getPhoneNumber());

            if ($userExists) {
                $this->responseService->addError('Пользователь уже существует');
            } else {
                try {
                    $user->setDeleted(false);
                    $user->setConfirm(false);
                    $hashedPassword = $this->passwordHasher->hashPassword(
                        $user,
                        $dto->password ?: ''
                    );
                    $user->setPassword($hashedPassword);

                    $em = $this->doctrine->getManager();

                    $em->persist($user);
                    $em->flush();
                    $this->responseService->addMessage('Пользователь зарегистрирован');

                    $this->sendService->setUser($user);
                    $this->sendService->setResponse($this->responseService);
                    $this->sendService->send();
                } catch (\Exception $exception) {
                    $this->responseService->addError('Ошибка регистрации пользователя');
                }
            }
        }
    }

    /**
     * Создание пользователя из Dto
     *
     * @return User|null
     */
    private function createUser(): ?User
    {
        $user = null;

        $data = $this->dtoService->toArray();

        if ($data) {
            $user = new User();

            $reflectionClass = new ReflectionClass($user);
            foreach ($reflectionClass->getProperties() as $property) {
                $type = $property->getType();
                if (isset($data[$property->getName()])) {
                    $sValue = $data[$property->getName()] ?: null;
                    if ($sValue !== null && $type !== null && ($type->getName() !== 'array')) {
                        $property->setValue($user, $sValue);
                    }
                }
            }
        } else {
            $this->responseService->addError('Ошибка получения данных');
        }

        return $user;
    }

    public function needDto(): bool
    {
        return true;
    }
}