<?php

declare(strict_types=1);

namespace App\Controller;

use App\Requests\RestaurantListRequest;
use Exception;
use App\Service\RestaurantService;
use Nelmio\ApiDocBundle\Annotation\Model;
use OpenApi\Attributes as OA;
use OpenApi\Attributes\Schema;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Attribute\Route;
use App\Model\RestaurantList;
use Symfony\Component\Serializer\SerializerInterface;

#[Route("/api/v1")]
class RestaurantController extends AbstractController
{
    public function __construct(private RestaurantService $restaurantService) {}

    #[Route('/restaurants', name: 'restaurants', methods: ['GET'])]
    #[OA\Response(response: 200, description: "Листинг ресторанов")]
    #[OA\Parameter(
         name: "page",
         description: "Номер страницы",
         in: "query",
         schema: new Schema(type: "integer", default: 1, example: 1)
    )]
    #[OA\Parameter(
        name: "limit",
        description: "Лимит",
        in: "query",
        schema: new Schema(type: "integer", default: 12, example: 12)
    )]
    #[OA\Parameter(
        name: "restaurant_type_id",
        description: "Идентификатор типа ресторанов",
        in: "query",
        schema: new Schema(type: "integer")
    )]
    #[OA\Parameter(
        name: "kitchen_id",
        description: "Идентификатор кухни",
        in: "query",
        schema: new Schema(type: "integer")
    )]
    #[Model(type: RestaurantList::class)]
    public function restaurants(RestaurantListRequest $request): Response
    {
        $page = $request->getRequest()->query->get('page');
        $limit = $request->getRequest()->query->get('limit');
        $restaurantTypeId = $request->getRequest()->query->get('restaurant_type_id');
        $kitchenId = $request->getRequest()->query->get('kitchen_id');
        $restaurantsList = $this->restaurantService->getRestaurants(
            $page, $limit, $restaurantTypeId, $kitchenId
        );
        return $this->json($restaurantsList);
    }

    #[Route('/restaurants/{restaurantId}', name: 'restaurant', methods: ['GET'])]
    public function restaurant(Request $request, SerializerInterface $serializer): Response
    {
        try {
            $restaurantId = (int)$request->get('restaurantId');
            $restaurant = $this->restaurantService->getRestaurant($restaurantId);
            $json = $serializer->serialize($restaurant, 'json', [
                'circular_reference_handler' => function ($object) {
                    return $object->getId();
                }
            ]);
            return new Response($json, Response::HTTP_OK, ['Content-Type' => 'application/json']);
        } catch (Exception $e) {
            return new Response($e->getMessage(), $e->getCode());
        }
    }
}
