<?php

namespace App\Service;

use App\Entity\File;
use App\Entity\Gallery;
use App\Entity\Kitchen;
use App\Entity\Restaurant;
use App\Entity\RestaurantType;
use App\Entity\Seo;
use App\Entity\Settlement;
use App\Exception\FileNotFoundException;
use App\Exception\RestaurantNotFoundException;
use App\Exception\RestaurantTypeNotFoundException;
use App\Exception\SeoNotFoundException;
use App\Exception\SettlementNotFoundException;
use App\Repository\FileRepository;
use App\Repository\Interface\GalleryRepositoryInterface;
use App\Repository\SeoRepository;
use App\Repository\SettlementRepository;
use App\Requests\CreateRestaurantRequest;
use App\Requests\EditRestaurantRequest;
use App\Requests\RestaurantListRequest;
use App\Mapper\RestaurantMapper;
use App\Model\RestaurantDetailElement;
use App\Model\RestaurantList;
use App\Repository\Interface\KitchenRepositoryInterface;
use App\Repository\Interface\RestaurantRepositoryInterface;
use App\Repository\Interface\RestaurantTypeRepositoryInterface;
use Ramsey\Collection\Collection;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Uid\Uuid;

class RestaurantService
{
    private const DEFAULT_PAGE  = 1;
    private const DEFAULT_LIMIT = 12;

    public function __construct(
        private RestaurantRepositoryInterface $restaurantRepository,
        private RestaurantTypeRepositoryInterface $restaurantTypeRepository,
        private KitchenRepositoryInterface $kitchenRepository,
        private GalleryRepositoryInterface $galleryRepository,
        private SettlementRepository $settlementRepository,
        private SeoRepository $seoRepository,
        private FIleRepository $fileRepository,
    ) {}

    public function getRestaurantsByRequest(
        RestaurantListRequest $request
    ): RestaurantList {
        $page = $request->getRequest()->query->get('page') ?? self::DEFAULT_PAGE;
        $limit = $request->getRequest()->query->get('limit') ?? self::DEFAULT_LIMIT;
        $restaurantTypeId = $request->getRequest()->query->get('restaurant_type_id');
        $kitchenId = $request->getRequest()->query->get('kitchen_id');

        return $this->getRestaurants(
            $page, $limit, $restaurantTypeId, $kitchenId
        );
    }

    public function getRestaurants(
        int $page,
        int $limit,
        string|null $restaurantTypeId,
        string|null $kitchenId
    ): RestaurantList {
        $restaurants = new Collection(
            Restaurant::class,
            $this->restaurantRepository->getAll(
                $page,
                $limit,
                $restaurantTypeId,
                $kitchenId
            )
        );
        $count = $this->restaurantRepository->getCount();
        $restaurantTypes = new Collection(
            RestaurantType::class,
            $this->restaurantTypeRepository->getAll()
        );
        $kitchens = new Collection(
            Kitchen::class,
            $this->kitchenRepository->getAll()
        );

        return RestaurantMapper::mapToRestaurantList(
            $restaurants,
            $restaurantTypes,
            $kitchens,
            $page,
            $limit,
            $count
        );
    }

    public function getRestaurantByRequest(Request $request): RestaurantDetailElement
    {
        return $this->getRestaurant(
            $request->get('restaurantId')
        );
    }

    public function getRestaurant(string $id): RestaurantDetailElement
    {
        $restaurant = $this->restaurantRepository->getById($id);
        if ($restaurant === null) {
            throw new RestaurantNotFoundException();
        }

        $gallery = new Collection(
            Gallery::class,
            $this->galleryRepository->getGalleryByRestaurantId($restaurant->getId())
        );

        return RestaurantMapper::mapToDetailElement($restaurant, $gallery);
    }

    public function addRestaurantByRequest(
        CreateRestaurantRequest|EditRestaurantRequest $request,
        string $id = null
    ): RestaurantDetailElement {
        $restaurantType = $this->restaurantTypeRepository->getById($request->getTypeId());
        $settlement = $this->settlementRepository->getById($request->getSettlementId());
        $seo = $this->seoRepository->getById($request->getSeoId());
        $file = $this->fileRepository->getById($request->getFileId());
        $this->checkForNulls($restaurantType, $settlement, $seo, $file);

        $restaurant = $this->restaurantRepository->create(
            RestaurantMapper::createRestaurantEntity(
                $request,
                $restaurantType,
                $settlement,
                $seo,
                $file,
                $id
        ));

        return RestaurantMapper::mapToDetailElement($restaurant, null);
    }

    public function putRestaurantByRequest(
        string $id,
        EditRestaurantRequest $request
    ): RestaurantDetailElement {
        if ($this->restaurantRepository->restaurantExists($id)) {
            return $this->patchRestaurantByRequest($id, $request);
        }

        return $this->addRestaurantByRequest($request, $id);
    }

    public function patchRestaurantByRequest(
        string $id,
        EditRestaurantRequest $request
    ): RestaurantDetailElement {
        $restaurantType = $this->restaurantTypeRepository->getById($request->getTypeId());
        $settlement = $this->settlementRepository->getById($request->getSettlementId());
        $seo = $this->seoRepository->getById($request->getSeoId());
        $file = $this->fileRepository->getById($request->getFileId());
        $this->checkForNulls($restaurantType, $settlement, $seo, $file);

        if (!$this->restaurantRepository->restaurantExists($id)) {
            throw new RestaurantNotFoundException();
        }

        $newRestaurant = RestaurantMapper::updateRestaurantEntity(
            $request,
            $restaurantType,
            $settlement,
            $seo,
            $file,
            $this->restaurantRepository->getById($id)
        );

        return RestaurantMapper::mapToDetailElement(
            $this->restaurantRepository->update($newRestaurant),
            null
        );
    }

    public function deleteRestaurant(string $id): void
    {
        if (!$this->restaurantRepository->restaurantExists($id)) {
            throw new RestaurantNotFoundException();
        }

        $this->restaurantRepository->delete($id);
    }

    private function checkForNulls(
        ?RestaurantType $type,
        ?Settlement $settlement,
        ?Seo $seo,
        ?File $file
    ): void {
        if ($type === null) {
            throw new RestaurantTypeNotFoundException();
        }
        if ($settlement === null) {
            throw new SettlementNotFoundException();
        }
        if ($seo === null) {
            throw new SeoNotFoundException;
        }
        if ($file === null) {
            throw new FileNotFoundException();
        }
    }
}