<?php

namespace App\Service;

use App\Entity\File;
use App\Entity\News;
use App\Entity\NewsCategory;
use App\Entity\NewsType;
use App\Entity\Seo;
use App\Exception\FileNotFoundException;
use App\Exception\NewsNotFoundException;
use App\Exception\NewsTypeNotFoundException;
use App\Exception\SeoNotFoundException;
use App\Mapper\NewsMapper;
use App\Model\NewsDetailElement;
use App\Model\NewsList;
use App\Model\NewsListingElement;
use App\Repository\FileRepository;
use App\Repository\Interface\NewsCategoryRepositoryInterface;
use App\Repository\Interface\NewsRepositoryInterface;
use App\Repository\NewsTypeRepository;
use App\Repository\SeoRepository;
use App\Requests\CreateNewsRequest;
use App\Requests\EditNewsRequest;
use App\Requests\NewsListRequest;
use Ramsey\Collection\Collection;
use Symfony\Component\HttpFoundation\Request;

class NewsService
{
    private const DEFAULT_PAGE = 1;
    private const DEFAULT_LIMIT = 12;

    public function __construct(
        private NewsRepositoryInterface $newsRepository,
        private NewsCategoryRepositoryInterface $newsCategoryRepository,
        private NewsTypeRepository $newsTypeRepository,
        private SeoRepository $seoRepository,
        private FileRepository $fileRepository
    ) {}

    public function getNewsByRequest(NewsListRequest $request): NewsList
    {
        $page = $request->getRequest()->query->get('page') ?? self::DEFAULT_PAGE;
        $limit = $request->getRequest()->query->get('limit') ?? self::DEFAULT_LIMIT;
        $newsCategory = $request->getRequest()->query->get('news_category');

        return $this->getNews($page, $limit, $newsCategory);
    }

    public function getNews(int $page, int $limit, string|null $newsCategory): NewsList
    {
        $news = new Collection(
            News::class,
            $this->newsRepository->getAll(
                $page,
                $limit,
                $newsCategory
            )
        );
        $newsCategories = new Collection(
            NewsCategory::class,
            $this->newsCategoryRepository->getAll()
        );
        $count = $this->newsRepository->getCount();

        return NewsMapper::mapToNewsList($news, $newsCategories, $page, $limit, $count);
    }

    public function getMainNews(): NewsListingElement
    {
        return NewsMapper::mapToListingElement(
            $this->newsRepository->getMainNews()
        );
    }

    public function getNewsSearch(): NewsDetailElement
    {
        return NewsMapper::mapToDetailElement(
            $this->newsRepository->getMainNews()
        );
    }

    public function getNewsOneByRequest(Request $request): NewsDetailElement
    {
        return $this->getNewsOne(
            $request->get('newsId')
        );
    }

    public function getNewsOne(string $newsId): NewsDetailElement
    {
        $news = $this->newsRepository->getNewsById($newsId);
        if ($news === null) {
            throw new NewsNotFoundException();
        }

        return NewsMapper::mapToDetailElement($news);
    }

    public function addNewsByRequest(
        CreateNewsRequest|EditNewsRequest $request,
        string $id = null
    ): NewsDetailElement {
        $newsType = $this->newsTypeRepository->getById($request->getTypeId());
        $seo = $this->seoRepository->getById($request->getSeoId());
        $file = $this->fileRepository->getById($request->getFileId());
        $this->checkForNulls($newsType, $seo, $file);

        $news = $this->newsRepository->create(
            NewsMapper::createNewsEntity(
                $request,
                $newsType,
                $seo,
                $file,
                $id
            ));

        return NewsMapper::mapToDetailElement($news);
    }

    public function putNewsByRequest(
        string $id,
        EditNewsRequest $request
    ): NewsDetailElement {
        if ($this->newsRepository->newsExists($id)) {
            return $this->patchNewsByRequest($id, $request);
        }

        return $this->addNewsByRequest($request, $id);
    }

    public function patchNewsByRequest(
        string $id,
        EditNewsRequest $request
    ): NewsDetailElement {
        $newsType = $this->newsTypeRepository->getById($request->getTypeId());
        $seo = $this->seoRepository->getById($request->getSeoId());
        $file = $this->fileRepository->getById($request->getFileId());
        $this->checkForNulls($newsType, $seo, $file);

        if (!$this->newsRepository->newsExists($id)) {
            throw new NewsNotFoundException();
        }

        $newNews = NewsMapper::updateNewsEntity(
            $request,
            $newsType,
            $seo,
            $file,
            $this->newsRepository->getById($id)
        );

        return NewsMapper::mapToDetailElement(
            $this->newsRepository->update($newNews),
        );
    }

    public function deleteNews(string $id): void
    {
        if (!$this->newsRepository->newsExists($id)) {
            throw new NewsNotFoundException();
        }

        $this->newsRepository->delete($id);
    }

    private function checkForNulls(
        ?NewsType $type,
        ?Seo $seo,
        ?File $file
    ): void {
        if ($type === null) {
            throw new NewsTypeNotFoundException();
        }
        if ($seo === null) {
            throw new SeoNotFoundException;
        }
        if ($file === null) {
            throw new FileNotFoundException();
        }
    }
}