<?php

namespace App\Repository;

use App\Entity\Restaurant;
use App\Repository\Interface\RestaurantRepositoryInterface;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Component\Uid\Uuid;

/**
 * @extends ServiceEntityRepository<Restaurant>
 *
 * @method Restaurant|null find($id, $lockMode = null, $lockVersion = null)
 * @method Restaurant|null findOneBy(array $criteria, array $orderBy = null)
 * @method Restaurant[]    findAll()
 * @method Restaurant[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class RestaurantRepository extends ServiceEntityRepository implements RestaurantRepositoryInterface
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Restaurant::class);
    }

    public function getAll(
        int $page,
        int $limit,
        string|null $restaurantTypeId,
        string|null $kitchenId
    ): array {
        $query = $this->createQueryBuilder('r');
        $query->select('r');

        if ($restaurantTypeId !== null) {
            $query->andWhere('r.typeId = :restaurantTypeId')
                ->setParameter('restaurantTypeId', $restaurantTypeId);
        }
        if ($kitchenId !== null) {
            $query->join('r.kitchen', 'k')
                ->andWhere('k.id = :kitchenId')
                ->setParameter('kitchenId', $kitchenId);
        }

        $query->setMaxResults($limit)
            ->setFirstResult(($page - 1) * $limit);

        return $query->getQuery()->getResult();
    }

    public function getCount(): int
    {
        return $this->count();
    }

    public function getById(string $id): Restaurant|null
    {
        try {
            $uuid = new Uuid($id);
            return $this->find($uuid);
        } catch (\Exception $e) {
            return null;
        }
    }

    public function create(Restaurant $restaurant): Restaurant
    {
        $id = $restaurant->getId();
        $this->getEntityManager()->persist($restaurant);

        if ($id !== null) {
            $restaurant->setId($id);
        }

        $this->getEntityManager()->flush();
        return $restaurant;
    }

    public function update(Restaurant $restaurant): Restaurant
    {
        $this->getEntityManager()->persist($restaurant);
        $this->getEntityManager()->flush();
        return $restaurant;
    }

    public function restaurantExists(string $id): bool
    {
        return null !== $this->find(new Uuid($id));
    }

    public function delete(string $id): void
    {
        $this->getEntityManager()->remove($this->find(new Uuid($id)));
        $this->getEntityManager()->flush();
    }
}
