<?php

namespace App\Repository;

use Exception;
use App\Entity\News;
use App\Repository\Interface\NewsRepositoryInterface;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Component\Uid\Uuid;

/**
 * @extends ServiceEntityRepository<News>
 *
 * @method News|null find($id, $lockMode = null, $lockVersion = null)
 * @method News|null findOneBy(array $criteria, array $orderBy = null)
 * @method News[]    findAll()
 * @method News[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class NewsRepository extends ServiceEntityRepository implements NewsRepositoryInterface
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, News::class);
    }

    public function getAll(int $page, int $limit, string|null $newsCategory): array
    {
        $query = $this->getEntityManager()->createQueryBuilder();
        $query->select('n')->from(News::class, 'n');
        if ($newsCategory !== null) {
            $query->join('n.newsCategories', 'nc')
                ->andWhere('nc.id = :newsCategory')
                ->setParameter('newsCategory', $newsCategory);
        }
        $query->setMaxResults($limit)
            ->setFirstResult(($page - 1) * $limit);
        return $query->getQuery()->getResult();
    }

    public function getCount(): int
    {
        return $this->count();
    }

    public function getMainNews(): News
    {
        return $this->findOneBy(['mainPageRender' => true]);
    }

    public function getNewsById(string $newsId): News|null
    {
        try {
            $uuid = new Uuid($newsId);
            return $this->find($uuid);
        } catch (Exception $e) {
            return null;
        }
    }

    public function getById(string $id): News|null
    {
        try {
            $uuid = new Uuid($id);
            return $this->find($uuid);
        } catch (\Exception $e) {
            return null;
        }
    }

    public function create(News $news): News
    {
        $id = $news->getId();
        $this->getEntityManager()->persist($news);

        if ($id !== null) {
            $news->setId($id);
        }

        $this->getEntityManager()->flush();
        return $news;
    }

    public function update(News $news): News
    {
        $this->getEntityManager()->persist($news);
        $this->getEntityManager()->flush();
        return $news;
    }

    public function newsExists(string $id): bool
    {
        return null !== $this->find(new Uuid($id));
    }

    public function delete(string $id): void
    {
        $this->getEntityManager()->remove($this->find(new Uuid($id)));
        $this->getEntityManager()->flush();
    }
}
