<?php

namespace App\Mapper;

use App\Entity\Gallery;
use App\Entity\Kitchen;
use App\Entity\Restaurant;
use App\Entity\RestaurantType;
use App\Entity\Seo;
use App\Entity\Settlement;
use App\Entity\Tags;
use App\Entity\File;
use App\Model\KitchenType;
use App\Model\Pagination;
use App\Model\RestaurantDetailElement;
use App\Model\RestaurantFilterVariants;
use App\Model\RestaurantList;
use App\Model\RestaurantListingElement;
use App\Model\RestaurantType as RestaurantTypeModel;
use App\Model\Tag;
use App\Model\File as FileModel;
use App\Requests\CreateRestaurantRequest;
use App\Requests\EditRestaurantRequest;
use Ramsey\Collection\Collection;
use DateTimeImmutable;
use Symfony\Component\Uid\Uuid;

class RestaurantMapper
{
    public static function mapToRestaurantList(
        Collection $restaurants,
        Collection $restaurantTypes,
        Collection $kitchens,
        int $page,
        int $limit,
        int $count
    ): RestaurantList {
        return new RestaurantList(
            new Pagination($page, ceil($count / $limit), $limit),
            new Collection(RestaurantListingElement::class, array_map(
                function (Restaurant $restaurant) {
                    return self::mapToListElement($restaurant);
                }, $restaurants->toArray())),
            new RestaurantFilterVariants(
                new Collection(
                    RestaurantTypeModel::class, array_map(
                        function (RestaurantType $restaurantType) {
                            return self::mapToRestaurantType($restaurantType);
                        }, $restaurantTypes->toArray()
                    )
                ),
                new Collection(
                    KitchenType::class, array_map(
                        function (Kitchen $kitchen) {
                            return self::mapToKitchenType($kitchen);
                        }, $kitchens->toArray()
                    ),
                )
            )
        );
    }

    public static function mapToListElement(
        Restaurant $restaurant
    ): RestaurantListingElement {
        return new RestaurantListingElement(
            $restaurant->getId(),
            $restaurant->getName(),
            $restaurant->getCode(),
            self::mapToRestaurantType($restaurant->getTypeId()),
            $restaurant->getCheckInfo(),
            FileMapper::mapToFile($restaurant->getFile()),
            $restaurant->getSite()
        );
    }

    public static function mapToDetailElement(
        Restaurant $restaurant,
        Collection|null $gallery
    ): RestaurantDetailElement {
        return new RestaurantDetailElement(
            $restaurant->getId(),
            $restaurant->getName(),
            $restaurant->getCode(),
            implode(',', $restaurant->getCoordinates()),
            self::mapToRestaurantType($restaurant->getTypeId()),
            $restaurant->getCheckPrice(),
            $restaurant->getCheckInfo(),
            new Collection(
                KitchenType::class, array_map(
                function (Kitchen $kitchen) {
                    return self::mapToKitchenType($kitchen);
                }, $restaurant->getKitchen()->toArray()),
            ),
            new Collection("string", $restaurant->getPhone()),
            new Collection("string", $restaurant->getEmail()),
            new Collection("string", $restaurant->getAddress()),
            self::mapToTag(
                new Collection(
                    Tags::class,
                    $restaurant->getTags()->toArray()
                )
            ),
            $restaurant->getSite(),
            FileMapper::mapToFile($restaurant->getFile()),
            $gallery
                ? self::mapToGallery($gallery)
                : new Collection(FileModel::class),
            $restaurant->getSeo()?->getTitle(),
            $restaurant->getSeo()?->getDescription(),
            $restaurant->getSeo()?->getKeywords()
        );
    }

    public static function mapToRestaurantType(
        RestaurantType $restaurantType
    ): RestaurantTypeModel {
        return new RestaurantTypeModel(
            $restaurantType->getId(),
            $restaurantType->getName(),
            $restaurantType->getCode()
        );
    }

    public static function mapToKitchenType(Kitchen $kitchen): KitchenType
    {
        return new KitchenType(
            $kitchen->getId(),
            $kitchen->getName()
        );
    }

    public static function mapToTag(Collection $tags): Collection
    {
        return new Collection(Tag::class, array_map(
            function (Tags $tag) {
                return new Tag(
                    $tag->getName(),
                    new Collection("string", $tag->getList())
                );
            }, $tags->toArray())
        );
    }

    public static function mapToGallery(Collection $gallery): Collection
    {
        return new Collection(FileModel::class, array_map(
            function (Gallery $galleryOne) {
                return FileMapper::mapToFile($galleryOne->getFile());
            }, $gallery->toArray())
        );
    }

    public static function createRestaurantEntity(
        CreateRestaurantRequest|EditRestaurantRequest $request,
        RestaurantType $restaurantType,
        Settlement $settlement,
        Seo $seo,
        File $file,
        string $id = null
    ): Restaurant {
        $restaurant = (new Restaurant())
            ->setName($request->getName())
            ->setCode($request->getCode())
            ->setActive($request->isActive())
            ->setSort($request->getSort())
            ->setCreateAt(new DateTimeImmutable())
            ->setUpdateAt(new DateTimeImmutable())
            ->setCoordinates($request->getCoordinates())
            ->setTypeId($restaurantType)
            ->setSettlementId($settlement)
            ->setDescription($request->getDescription())
            ->setCheckPrice($request->getCheck())
            ->setCheckInfo($request->getCheckInfo())
            ->setPhone($request->getPhone())
            ->setEmail($request->getEmail())
            ->setAddress($request->getAddress())
            ->setSite($request->getSite())
            ->setPreviewImage($request->getPreviewImage())
            ->setDetailImage($request->getDetailImage())
            ->setSeo($seo)
            ->setFile($file)
            ->setHowToFind($request->getHowToFind());

        if ($id !== null) {
            $restaurant->setId(new Uuid($id));
        }

        return $restaurant;
    }

    public static function updateRestaurantEntity(
        EditRestaurantRequest $request,
        RestaurantType $restaurantType,
        Settlement $settlement,
        Seo $seo,
        File $file,
        Restaurant $restaurant
    ): Restaurant
    {
        $restaurant->setName($request->getName());
        $restaurant->setCode($request->getCode());
        $restaurant->setActive($request->isActive());
        $restaurant->setSort($request->getSort());
        $restaurant->setCreateAt($restaurant->getCreateAt());
        $restaurant->setUpdateAt(new DateTimeImmutable());
        $restaurant->setCoordinates($request->getCoordinates());
        $restaurant->setTypeId($restaurantType);
        $restaurant->setSettlementId($settlement);
        $restaurant->setDescription($request->getDescription());
        $restaurant->setCheckPrice($request->getCheck());
        $restaurant->setCheckInfo($request->getCheckInfo());
        $restaurant->setPhone($request->getPhone());
        $restaurant->setEmail($request->getEmail());
        $restaurant->setAddress($request->getAddress());
        $restaurant->setSite($request->getSite());
        $restaurant->setPreviewImage($request->getPreviewImage());
        $restaurant->setHowToFind($request->getHowToFind());
        $restaurant->setDetailImage($request->getDetailImage());
        $restaurant->setSeo($seo);
        $restaurant->setFile($file);
        return $restaurant;
    }
}