<?php

namespace App\Mapper;

use App\Entity\File;
use App\Entity\News;
use App\Entity\NewsCategory;
use App\Entity\NewsType;
use App\Entity\Seo;
use App\Model\NewsDetailElement;
use App\Model\NewsFilterVariants;
use App\Model\NewsList;
use App\Model\NewsListingElement;
use App\Model\NewsCategory as NewsCategoryModel;
use App\Model\Pagination;
use App\Requests\CreateNewsRequest;
use App\Requests\EditNewsRequest;
use Ramsey\Collection\Collection;
use DateTimeImmutable;
use Symfony\Component\Uid\Uuid;

class NewsMapper
{
    public static function mapToNewsList(
        Collection $news,
        Collection $newsCategory,
        int $page,
        int $limit,
        int $count
    ): NewsList {
        return new NewsList(
            new Pagination($page, ceil($count / $limit), $limit),
            new Collection(NewsListingElement::class, array_map(
                function (News $newsOne) {
                    return self::mapToListingElement($newsOne);
                }, $news->toArray())
            ),
            new NewsFilterVariants(
                new Collection(NewsCategoryModel::class, array_map(
                    function (NewsCategory $newsCategoryOne) {
                        return self::mapToNewsCategory($newsCategoryOne);
                    }, $newsCategory->toArray())
                )
            )
        );
    }

    public static function mapToListingElement(News $newsOne): NewsListingElement
    {
        return new NewsListingElement(
            $newsOne->getId(),
            $newsOne->getPreviewText(),
            $newsOne->getDetailText(),
            FileMapper::mapToFile($newsOne->getFile()),
            $newsOne->getCreateAt()?->format('d.m.Y'),
            "/{$newsOne->getCode()}/code/"
        );
    }

    public static function mapToDetailElement(News $newsOne): NewsDetailElement
    {
        return new NewsDetailElement(
            $newsOne->getId(),
            $newsOne->getPreviewText(),
            $newsOne->getDetailText(),
            $newsOne->getDetailText(),
            FileMapper::mapToFile($newsOne->getFile()),
            $newsOne->getCreateAt()?->format('d.m.Y'),
            $newsOne->getSeo()?->getTitle(),
            $newsOne->getSeo()?->getDescription(),
            $newsOne->getSeo()?->getKeywords()
        );
    }

    public static function mapToNewsCategory(
        NewsCategory $newsCategory
    ): NewsCategoryModel {
        return new NewsCategoryModel(
            $newsCategory->getId(),
            $newsCategory->getName(),
            $newsCategory->getCode(),
        );
    }

    public static function createNewsEntity(
        CreateNewsRequest|EditNewsRequest $request,
        NewsType $newsType,
        Seo $seo,
        File $file,
        string $id = null
    ): News {
        $news = (new News())
            ->setCode($request->getCode())
            ->setActive($request->isActive())
            ->setCreateAt(new DateTimeImmutable())
            ->setUpdateAt(new DateTimeImmutable())
            ->setSort($request->getSort())
            ->setPreviewImage($request->getPreviewImage())
            ->setDetailImage($request->getDetailImage())
            ->setPreviewText($request->getPreviewText())
            ->setDetailText($request->getDetailText())
            ->setType($newsType)
            ->setMainPageRender($request->isMainPageRender())
            ->setSeo($seo)
            ->setFile($file);

        if ($id !== null) {
            $news->setId(new Uuid($id));
        }

        return $news;
    }

    public static function updateNewsEntity(
        CreateNewsRequest|EditNewsRequest $request,
        NewsType $newsType,
        Seo $seo,
        File $file,
        News $news
    ): News
    {
        $news->setCode($request->getCode());
        $news->setActive($request->isActive());
        $news->setCreateAt($news->getCreateAt());
        $news->setUpdateAt(new DateTimeImmutable());
        $news->setSort($request->getSort());
        $news->setPreviewText($request->getPreviewText());
        $news->setDetailText($request->getDetailText());
        $news->setPreviewImage($request->getPreviewImage());
        $news->setDetailImage($request->getDetailImage());
        $news->setType($newsType);
        $news->setMainPageRender($request->isMainPageRender());
        $news->setSeo($seo);
        $news->setFile($file);

        return $news;
    }
}