<?php

declare(strict_types=1);

namespace App\Controller;

use App\Attribute\RequestBody;
use App\Exception\RestaurantNotFoundException;
use App\Requests\CreateRestaurantRequest;
use App\Requests\EditRestaurantRequest;
use App\Requests\RestaurantListRequest;
use App\Service\RestaurantService;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Attribute\Route;

#[Route("/api/v1")]
class RestaurantController extends AbstractController
{
    public function __construct(private RestaurantService $restaurantService) {}

    #[Route('/restaurants', name: 'restaurants', methods: ['GET'])]
    public function restaurants(RestaurantListRequest $request): Response
    {
        return $this->json(
            $this->restaurantService->getRestaurantsByRequest($request)
        );
    }

    #[Route('/restaurants/{restaurantId}', name: 'restaurant', methods: ['GET'])]
    public function restaurant(Request $request): Response
    {
        try {
            return $this->json(
                $this->restaurantService->getRestaurantByRequest($request),
            );
        } catch (RestaurantNotFoundException $e) {
            return $this->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], $e->getCode());
        }
    }

    #[Route('/restaurant', name: 'addRestaurant', methods: ['POST'])]
    public function addRestaurant(
        #[RequestBody] CreateRestaurantRequest $request
    ): Response {
        return $this->json($this->restaurantService->addRestaurantByRequest($request));
    }

    #[Route('/restaurant/{id}', name: 'putRestaurant', methods: ['PUT'])]
    public function putRestaurant(
        string $id,
        #[RequestBody] EditRestaurantRequest $request
    ): Response {
        return $this->json($this->restaurantService->putRestaurantByRequest($id, $request));
    }

    #[Route('/restaurant/{id}', name: 'patchRestaurant', methods: ['PATCH'])]
    public function patchRestaurant(
        string $id,
        #[RequestBody] EditRestaurantRequest $request
    ): Response {
        return $this->json($this->restaurantService->patchRestaurantByRequest($id, $request));
    }

    #[Route('/restaurant/{id}', name: 'deleteRestaurant', methods: ['DELETE'])]
    public function deleteRestaurant(string $id): Response
    {
        $this->restaurantService->deleteRestaurant($id);
        return $this->json(null);
    }
}
