<?php

declare(strict_types=1);

namespace App\Controller;

use App\Attribute\RequestBody;
use App\Exception\NewsNotFoundException;
use App\Requests\CreateNewsRequest;
use App\Requests\EditNewsRequest;
use App\Requests\NewsListRequest;
use App\Service\NewsService;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Attribute\Route;

#[Route("/api/v1")]
class NewsController extends AbstractController
{
    public function __construct(private NewsService $newsService) {}

    #[Route('/news', name: 'news', methods: ['GET'])]
    public function news(NewsListRequest $request): Response
    {
        return $this->json(
            $this->newsService->getNewsByRequest($request)
        );
    }

    #[Route('/news/mainNews', name: 'mainNews', methods: ['GET'])]
    public function mainNews(): Response
    {
        return $this->json(
            $this->newsService->getMainNews()
        );
    }

    #[Route('/news/search', name: 'newsSearch', methods: ['GET'])]
    public function newsSearch(): Response
    {
        return $this->json(
            $this->newsService->getNewsSearch()
        );
    }

    #[Route('/news/{newsId}', name: 'newsOne', methods: ['GET'])]
    public function newsOne(Request $request): Response
    {
        try {
            return $this->json(
                $this->newsService->getNewsOneByRequest($request)
            );
        } catch (NewsNotFoundException $e) {
            return $this->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], $e->getCode());
        }
    }

    #[Route('/news', name: 'addNews', methods: ['POST'])]
    public function addNews(
        #[RequestBody] CreateNewsRequest $request
    ): Response {
        return $this->json($this->newsService->addNewsByRequest($request));
    }

    #[Route('/news/{id}', name: 'putNews', methods: ['PUT'])]
    public function putNews(
        string $id,
        #[RequestBody] EditNewsRequest $request
    ): Response {
        return $this->json($this->newsService->putNewsByRequest($id, $request));
    }

    #[Route('/news/{id}', name: 'patchNews', methods: ['PATCH'])]
    public function patchNews(
        string $id,
        #[RequestBody] EditNewsRequest $request
    ): Response {
        return $this->json($this->newsService->patchNewsByRequest($id, $request));
    }

    #[Route('/news/{id}', name: 'deleteNews', methods: ['DELETE'])]
    public function deleteNews(string $id): Response
    {
        $this->newsService->deleteNews($id);
        return $this->json(null);
    }
}
