<?php

namespace App\Mapper;

use App\Entity\Address;
use App\Entity\Email;
use App\Entity\Kitchen;
use App\Entity\Phone;
use App\Entity\Restaurant;
use App\Entity\RestaurantType;
use App\Entity\Tags;
use App\Model\File;
use App\Model\KitchenType;
use App\Model\Pagination;
use App\Model\RestaurantDetailElement;
use App\Model\RestaurantFilterVariants;
use App\Model\RestaurantList;
use App\Model\RestaurantListingElement;
use App\Model\RestaurantType as RestaurantTypeModel;
use App\Model\Tag;
use Ramsey\Collection\Collection;

class RestaurantMapper
{
    public static function mapToRestaurantList(
        $restaurants,
        $restaurantTypes,
        $kitchens,
        $page,
        $limit,
        $count): RestaurantList
    {
        return new RestaurantList(
            new Pagination($page, ceil($count / $limit), $limit),
            new Collection(RestaurantListingElement::class, array_map(
                function (Restaurant $restaurant) {
                    return RestaurantMapper::mapToListElement($restaurant);
                }, $restaurants)),
            new RestaurantFilterVariants(
                new Collection(
                    RestaurantTypeModel::class, array_map(
                        function (RestaurantType $restaurantType) {
                            return RestaurantMapper::mapToRestaurantType($restaurantType);
                        }, $restaurantTypes
                    )
                ),
                new Collection(
                    KitchenType::class, array_map(
                    function (Kitchen $kitchen) {
                        return RestaurantMapper::mapToKitchenType($kitchen);
                    }, $kitchens
                ),
                )
            )
        );
    }

    public static function mapToListElement(Restaurant $restaurant): RestaurantListingElement
    {
        return new RestaurantListingElement(
            $restaurant->getId(),
            $restaurant->getName(),
            $restaurant->getCode(),
            self::mapToRestaurantType($restaurant->getTypeId()),
            $restaurant->getCheckInfo(),
            new File(
                1,
                "name",
                "description",
                10,
                "jpg",
                $restaurant->getPreviewImage()
            ),
            $restaurant->getSite()
        );
    }

    public static function mapToDetailElement(Restaurant $restaurant): RestaurantDetailElement
    {
        $file = new File(
            1,
            "name",
            "description",
            10,
            "jpg",
            $restaurant->getPreviewImage()
        );
        return new RestaurantDetailElement(
            $restaurant->getId(),
            $restaurant->getName(),
            $restaurant->getCode(),
            implode(',', $restaurant->getCoordinates()),
            self::mapToRestaurantType($restaurant->getTypeId()),
            $restaurant->getCheckPrice(),
            $restaurant->getCheckInfo(),
            new Collection(
                KitchenType::class, array_map(
                function (Kitchen $kitchen) {
                    return self::mapToKitchenType($kitchen);
                }, $restaurant->getKitchen()->toArray()),
            ),
            new Collection("string", array_map(
                function (Phone $phone) {
                    return self::mapToPhone($phone);
                }, $restaurant->getPhone()->toArray()
            )),
            new Collection("string", array_map(
                function (Email $email) {
                    return self::mapToEmail($email);
                }, $restaurant->getEmail()->toArray()
            )),
            new Collection("string", array_map(
                function (Address $address) {
                    return self::mapToAddress($address);
                }, $restaurant->getAddress()->toArray()
            )),
            new Collection(Tag::class, array_map(
                function (Tags $tag) {
                    return new Tag(
                        "группа тегов 1",
                        new Collection("string", [$tag->getName()]),
                    );
                }, $restaurant->getTags()->toArray()
            )),
            $restaurant->getSite(),
            $file,
            new Collection(File::class, [$file]),
            "Отель Арктика",
            "otel arktika",
            "otel arktika"
        );
    }

    public static function mapToRestaurantType(RestaurantType $restaurantType): RestaurantTypeModel
    {
        return new RestaurantTypeModel(
            $restaurantType->getId(),
            $restaurantType->getName(),
            $restaurantType->getCode()
        );
    }

    public static function mapToKitchenType(Kitchen $kitchen): KitchenType
    {
        return new KitchenType(
            $kitchen->getId(),
            $kitchen->getName()
        );
    }

    public static function mapToPhone(Phone $phone): string
    {
        return $phone->getName();
    }

    public static function mapToEmail(Email $email): string
    {
        return $email->getName();
    }

    public static function mapToAddress(Address $address): string
    {
        return $address->getName();
    }
}