<?php

namespace App\Tests\NewsTests\Controller;

use App\News\Collection\NewsCategoryCollection;
use App\News\Dto\NewsCategoryDto;
use App\News\Dto\NewsDetailElementDto;
use App\News\Dto\NewsFilterVariantsDto;
use App\News\Dto\NewsListDto;
use App\News\Dto\NewsListingElementDto;
use App\Shared\Collection\ListingCollection;
use App\Shared\Dto\FileDto;
use App\Shared\Dto\PaginationDto;
use DateTimeImmutable;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Symfony\Component\HttpFoundation\JsonResponse;

class NewsControllerTests extends WebTestCase
{
    // Тест oneNews
    public function testRightUuid(): void
    {
        $client = static::createClient();
        $client->request(
            'GET',
            '/api/v1/news/00000000-0000-0000-0000-000000000000'
        );


        self::assertResponseIsSuccessful();

        /** Проверка получемых новостей */
        self::assertSame(
            (new JsonResponse(
                new NewsDetailElementDto(
                    '00000000-0000-0000-0000-000000000000',
                    'news1',
                    'Preview text',
                    (new DateTimeImmutable())->format('d.m.Y'),
                    'Detail text',
                    new FileDto(
                        '00000000-0000-0000-0000-000000000000',
                        'file1',
                        'fileDescription',
                        1024,
                        'png',
                        '/somewhere/'
                    ),
                )
            ))->getContent(),
            $client->getResponse()->getContent(),
            'Полученная детальная новость инккоректена'
        );
    }

    public function testNonExistedUuid(): void
    {
        $client = static::createClient();
        $client->request(
            'GET',
            '/api/v1/news/00000000-0000-0000-0000-000000000005'
        );

        self::assertResponseStatusCodeSame(404);
    }

    public function testCorruptUuid(): void
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/news/looks-like-not-a-UUId');

        self::assertResponseStatusCodeSame(400);
    }


    // Тест news
    public function testNullRequest(): void
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/news/');

        self::assertResponseIsSuccessful();

        $testResponse = new JsonResponse(
            new NewsListDto(
                new PaginationDto(1, 1, 12),
                new ListingCollection(
                    [
                        new NewsListingElementDto(
                            '00000000-0000-0000-0000-000000000000',
                            'news1',
                            (new DateTimeImmutable())->format('d.m.Y'),
                            'api/v1/news/00000000-0000-0000-0000-000000000000',
                            'Preview text',
                            new FileDto(
                                '00000000-0000-0000-0000-000000000000',
                                'file1',
                                'fileDescription',
                                1024,
                                'png',
                                '/somewhere/'
                            ),
                        ),
                        new NewsListingElementDto(
                            '00000000-0000-0000-0000-000000000001',
                            'news2',
                            (new DateTimeImmutable())->format('d.m.Y'),
                            'api/v1/news/00000000-0000-0000-0000-000000000001',
                            null,
                            null,
                        ),
                    ],
                ),
                new NewsFilterVariantsDto(
                    new NewsCategoryCollection(
                        [
                            new NewsCategoryDto(
                                '00000000-0000-0000-0000-000000000000',
                                'newsCategory1',
                                'newsCategory_1',
                            ),
                            new NewsCategoryDto(
                                '00000000-0000-0000-0000-000000000001',
                                'newsCategory2',
                                'newsCategory_2',
                            ),
                        ]
                    ),
                ),
            ),
        );

        self::assertSame(
            $testResponse->getContent(),
            $client->getResponse()->getContent(),
            'Полученное тело респонза неверно'
        );
    }

    public function testNegativePagination(): void
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/news/?page=-10&limit=-10');
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();

        /** Проверка пагинации */
        self::assertStringContainsString(
            (new JsonResponse(new PaginationDto(1, 1, 12)))->getContent(),
            $data,
            'Пагинация инкоректна'
        );
    }

    public function testPageOverListCountPagination(): void
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/news/?page=14');
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();

        /** Проверка пагинации */
        self::assertStringContainsString(
            (new JsonResponse(new PaginationDto(1, 1, 12)))->getContent(),
            $data,
            'Пагинация инкоректна'
        );
    }

    public function testRightFilter(): void
    {
        $client = static::createClient();
        $client->request(
            'GET',
            '/api/v1/news/?news_category=00000000-0000-0000-0000-000000000000'
        );
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();


        /** Проверка получемых новостей */
        self::assertStringNotContainsString(
            (new JsonResponse(
                new ListingCollection(
                    [
                        new NewsListingElementDto(
                            '00000000-0000-0000-0000-000000000000',
                            'news1',
                            (new DateTimeImmutable())->format('d.m.Y'),
                            'api/v1/news/00000000-0000-0000-0000-000000000000',
                            'Preview text',
                            new FileDto(
                                '00000000-0000-0000-0000-000000000000',
                                'file1',
                                'fileDescription',
                                1024,
                                'png',
                                '/somewhere/',
                            ),
                        ),
                        new NewsListingElementDto(
                            '00000000-0000-0000-0000-000000000001',
                            'news2',
                            (new DateTimeImmutable())->format('d.m.Y'),
                            'api/v1/news/00000000-0000-0000-0000-000000000001',
                            null,
                            null,
                        ),
                    ]
                )
            ))->getContent(),
            $data,
            'Полученный список новостей инккоректен при правильном фильтре'
        );
    }

    public function testCorruptFilter(): void
    {
        $client = static::createClient();
        $client->request(
            'GET',
            '/api/v1/news/?news_category=looks-like-not-a-UUId'
        );
        $data = $client->getResponse()->getContent();

        self::assertResponseStatusCodeSame(200);
    }


    // mainNews
    public function testMainNews()
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/news/mainNews');
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();
    }

    public function testSearch()
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/news/search');
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();
    }
}
