<?php

namespace App\Shared\Repository;

use App\Shared\Entity\Restaurants;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use Doctrine\Persistence\ManagerRegistry;
use Ramsey\Collection\Collection;

/**
 * @extends ServiceEntityRepository<Restaurants>
 *
 * @method Restaurants|null find($id, $lockMode = null, $lockVersion = null)
 * @method Restaurants|null findOneBy(array $criteria, array $orderBy = null)
 * @method Restaurants[]    findAll()
 * @method Restaurants[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class RestaurantsRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Restaurants::class);
    }

    public function createUpdate(Restaurants $restaurants): void
    {
        $this->getEntityManager()->persist($restaurants);
        $this->getEntityManager()->flush();
        $this->getEntityManager()->clear();
    }

    public function remove(Restaurants $restaurants): void
    {
        $this->getEntityManager()->remove($restaurants);
        $this->getEntityManager()->flush();
        $this->getEntityManager()->clear();
    }

    public function getMaxValueSort(): ?int
    {
        return $this->createQueryBuilder('r')
            ->select('MAX(r.sort)')
            ->getQuery()
            ->getSingleScalarResult();
    }

    /** @return Collection<Restaurants> */
    protected function toCollection(QueryBuilder $query): Collection
    {
        return new Collection(
            Restaurants::class,
            $query
                ->orderBy('r.sort', 'ASC')
                ->getQuery()
                ->getResult()
        );
    }

    protected function filterByKitchen(?string $kitchenId, QueryBuilder $query): QueryBuilder
    {
        if ($kitchenId !== null) {
            $query = $query
                ->innerJoin('r.kitchens', 'k', Join::WITH, 'k.id = :kitchenId')
                ->setParameter('kitchenId', $kitchenId);
        }
        return $query;
    }

    protected function filterByType(?string $typeId, QueryBuilder $query): QueryBuilder
    {
        if ($typeId !== null) {
            $query = $query
                ->where('r.type = :typeId')
                ->setParameter('typeId', $typeId);
        }
        return $query;
    }

    public function getCountWithFilters(?string $kitchenId, ?string $typeId): int
    {
        $query = $this->createQueryBuilder('r');
        $query = $this->filterByKitchen($kitchenId, $query);
        $query = $this->filterByType($typeId, $query);
        return $query->select('COUNT(r.id)')->getQuery()->getSingleScalarResult();
    }

    /** @return Collection<Restaurants> */
    public function getWithFilters(?string $kitchenId, ?string $typeId, int $limit, int $offset): Collection
    {
        $query = $this->createQueryBuilder('r');
        $query = $this->filterByKitchen($kitchenId, $query);
        $query = $this->filterByType($typeId, $query);
        $query = $query->setMaxResults($limit);
        $query = $query->setFirstResult($offset);
        return $this->toCollection($query);
    }
}
