<?php

namespace App\Shared\EntityFactory;

use App\Restaurants\Request\RestaurantCreateRequest;
use App\Restaurants\Request\RestaurantFullUpdateRequest;
use App\Restaurants\Request\RestaurantPartUpdateRequest;
use App\Shared\Entity\Restaurants;
use App\Shared\Error\NotFoundError;
use App\Shared\Repository\FileRepository;
use App\Shared\Repository\KitchensRepository;
use App\Shared\Repository\RestaurantsRepository;
use App\Shared\Repository\RestaurantTypesRepository;
use App\Shared\Repository\SettlementsRepository;
use Ramsey\Collection\Collection;
use Ramsey\Uuid\Uuid;

class RestaurantEntityFactory
{
    /** @var Collection<callable> */
    private readonly Collection $restaurantSetters;

    public function __construct(
        private readonly RestaurantsRepository $restaurantsRepository,
        private readonly RestaurantTypesRepository $restaurantTypesRepository,
        private readonly SettlementsRepository $settlementsRepository,
        private readonly FileRepository $fileRepository,
        private readonly KitchensRepository $kitchensRepository,
    ) {
        $this->restaurantSetters = $this->setRestaurantSetters();
    }

    public function create(RestaurantCreateRequest $request): Restaurants
    {
        $restaurants = new Restaurants();
        $restaurants->setId(Uuid::uuid4()->toString());

        foreach ($request as $key => $value) {
            if ($value !== null) {
                $this->restaurantSetters[$key]($value, $restaurants);
            }
        }

        return $restaurants;
    }

    public function softUpdate(RestaurantFullUpdateRequest $request
    ): Restaurants {
        $restaurants = $this->restaurantsRepository->find(
            $request->id
        ) ?? new Restaurants();

        foreach ($request as $key => $value) {
            if ($value !== null) {
                $this->restaurantSetters[$key]($value, $restaurants);
            }
        }

        return $restaurants;
    }

    public function hardUpdate(RestaurantPartUpdateRequest $request
    ): Restaurants {
        $restaurant = $this->restaurantsRepository->find($request->id);

        if ($restaurant === null) {
            throw new NotFoundError('News entity for update not found');
        }

        foreach ($request as $key => $value) {
            if ($value !== null) {
                $this->restaurantSetters[$key]($value, $restaurant);
            }
        }

        return $restaurant;
    }

    /** @return Collection<callable> */
    private function setRestaurantSetters(): Collection
    {
        $setters = new Collection('callable');

        $setters['id'] = function (string $value, Restaurants $restaurant) {
            $restaurant->setId($value);
        };
        $setters['type_id'] = function (
            string $value,
            Restaurants $restaurant
        ) {
            $restaurant->setType(
                $this->restaurantTypesRepository->find($value)
            );
        };
        $setters['settlement_id'] = function (
            string $value,
            Restaurants $restaurant
        ) {
            $restaurant->setSettlement(
                $this->settlementsRepository->find($value)
            );
        };
        $setters['preview_image_id'] = function (
            string $value,
            Restaurants $restaurant
        ) {
            $restaurant->setPreviewImage($this->fileRepository->find($value));
        };
        $setters['detail_image_id'] = function (
            string $value,
            Restaurants $restaurant
        ) {
            $restaurant->setDetailImage($this->fileRepository->find($value));
        };
        $setters['active'] = function (bool $value, Restaurants $restaurant) {
            $restaurant->setActive($value);
        };
        $setters['name'] = function (string $value, Restaurants $restaurant) {
            $restaurant->setName($value);
        };
        $setters['code'] = function (string $value, Restaurants $restaurant) {
            $restaurant->setCode($value);
        };
        $setters['description'] = function (
            string $value,
            Restaurants $restaurant
        ) {
            $restaurant->setDescription($value);
        };
        $setters['check'] = function (string $value, Restaurants $restaurant) {
            $restaurant->setReceipt($value);
        };
        $setters['check_info'] = function (
            string $value,
            Restaurants $restaurant
        ) {
            $restaurant->setReceiptInfo($value);
        };
        $setters['phone'] = function (string $value, Restaurants $restaurant) {
            $restaurant->setPhone($value);
        };
        $setters['email'] = function (string $value, Restaurants $restaurant) {
            $restaurant->setEmail($value);
        };
        $setters['address'] = function (
            string $value,
            Restaurants $restaurant
        ) {
            $restaurant->setAddress($value);
        };
        $setters['tags'] = function (
            string $value,
            Restaurants $restaurant
        ) {
            $restaurant->setTags($value);
        };
        $setters['site'] = function (string $value, Restaurants $restaurant) {
            $restaurant->setSite($value);
        };
        $setters['coordinates'] = function (
            string $value,
            Restaurants $restaurant
        ) {
            $restaurant->setCoordinates($value);
        };
        $setters['how_to_find'] = function (
            string $value,
            Restaurants $restaurant
        ) {
            $restaurant->setHowToFind($value);
        };
        $setters['kitchens_id'] = function (
            Collection $ids,
            Restaurants $restaurant
        ) {
            foreach ($ids as $id) {
                $restaurant->addKitchen(
                    $this->kitchensRepository->find($id)
                );
            }
        };
        $setters['gallery_id'] = function (
            Collection $ids,
            Restaurants $restaurant
        ) {
            foreach ($ids as $id) {
                $restaurant->addGallery(
                    $this->fileRepository->find($id)
                );
            }
        };

        return $setters;
    }
}