<?php

namespace App\Requests;

use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Validator\Exception\ValidatorException;
use Symfony\Component\Validator\Validator\ValidatorInterface;

abstract class BaseRequest
{
    /**
     * активация автовалидации
     * перезаписать на false для отключение автовалидации
     */
    protected const AUTO_VALIDATE = true;

    public function __construct(protected ValidatorInterface $validator)
    {
        $this->populate();

        if (self::AUTO_VALIDATE) {
            $this->validate();
        }
    }

    protected function populate(): void
    {
        foreach ($this->getRequest()->toArray() as $property => $value) {
            if (property_exists($this, $property)) {
                $this->{$property} = $value;
            }
        }
    }

    /**
     * валидация и выброкса ошибки при валидации
     * @return void
     */
    public function validate()
    {
        $errors = $this->validator->validate($this);

        $messages = [
            'message' => 'validation_failed',
            'errors' => []
        ];

        foreach ($errors as $error) {
            $messages['errors'][] = [
                'property' => $error->getPropertyPath(),
                'value' => $error->getInvalidValue(),
                'message' => $error->getMessage(),
            ];
        }

        if (count($messages['errors']) > 0) {
            $response = new JsonResponse($messages, 201);
            $response->send();

            throw new ValidatorException('Validation failed', $messages);
        }
    }

    public function getRequest(): Request
    {
        return Request::createFromGlobals();
    }

    abstract public function serialise(): mixed;
}